/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.ast;

import com.google.inject.Inject;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.ast.ModelingElementExtensions;
import fr.inria.diverse.melange.codegen.ModelTypeGeneratorAdapterFactory;
import fr.inria.diverse.melange.eclipse.EclipseProjectHelper;
import fr.inria.diverse.melange.lib.MatchingHelper;
import fr.inria.diverse.melange.metamodel.melange.ExternalLanguage;
import fr.inria.diverse.melange.metamodel.melange.Language;
import fr.inria.diverse.melange.metamodel.melange.ModelType;
import java.io.IOException;
import java.util.List;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.codegen.ecore.generator.Generator;
import org.eclipse.emf.codegen.ecore.generator.GeneratorAdapterFactory;
import org.eclipse.emf.codegen.ecore.genmodel.GenJDKLevel;
import org.eclipse.emf.codegen.ecore.genmodel.GenModel;
import org.eclipse.emf.codegen.ecore.genmodel.GenModelFactory;
import org.eclipse.emf.codegen.ecore.genmodel.GenModelPackage;
import org.eclipse.emf.codegen.ecore.genmodel.GenPackage;
import org.eclipse.emf.codegen.ecore.genmodel.generator.GenBaseGeneratorAdapter;
import org.eclipse.emf.common.util.Diagnostic;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.Monitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.plugin.EcorePlugin;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * A collection of utilities around {@link ModelType}s
 */
@SuppressWarnings("all")
public class ModelTypeExtensions {
  @Inject
  @Extension
  private EclipseProjectHelper _eclipseProjectHelper;
  
  @Inject
  @Extension
  private IQualifiedNameProvider _iQualifiedNameProvider;
  
  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;
  
  @Inject
  @Extension
  private ModelingElementExtensions _modelingElementExtensions;
  
  @Inject
  private MatchingHelper matchingHelper;
  
  @Inject
  private EclipseProjectHelper helper;
  
  private final static Logger log = Logger.getLogger(ModelTypeExtensions.class);
  
  /**
   * Returns the URI of the serialized Ecore of the {@link ModelType}
   * {@code mt}.
   */
  public String getInferredEcoreUri(final ModelType mt) {
    final IProject project = this._eclipseProjectHelper.getProject(mt.eResource());
    if ((project == null)) {
      String _name = mt.getExtracted().getName();
      String _plus = ("Can\'t initialize exactType for " + _name);
      String _plus_1 = (_plus + " (Eclipse environment required)");
      ModelTypeExtensions.log.warn(_plus_1);
      return null;
    }
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("platform:/resource/");
    String _name_1 = project.getName();
    _builder.append(_name_1);
    _builder.append("/model-gen/");
    String _name_2 = mt.getName();
    _builder.append(_name_2);
    _builder.append(".ecore");
    return _builder.toString();
  }
  
  /**
   * Creates and serializes the {@link GenModel} for {@code mt} at the
   * location {@code gmUri}, pointing to the Ecore file located at {@code ecoreUri}.
   */
  public GenModel createGenmodel(final ModelType mt, final String ecoreUri, final String gmUri) {
    final ResourceSetImpl resSet = new ResourceSetImpl();
    resSet.getURIConverter().getURIMap().putAll(
      EcorePlugin.computePlatformURIMap(true));
    final Resource pkgRes = resSet.getResource(URI.createURI(ecoreUri), true);
    final Function1<EObject, EPackage> _function = new Function1<EObject, EPackage>() {
      @Override
      public EPackage apply(final EObject it) {
        return ((EPackage) it);
      }
    };
    final List<EPackage> pkgs = ListExtensions.<EObject, EPackage>map(pkgRes.getContents(), _function);
    final URI ecoreGmUri = EcorePlugin.getEPackageNsURIToGenModelLocationMap(true).get(EcorePackage.eNS_URI);
    final Resource ecoreGmRes = resSet.getResource(ecoreGmUri, true);
    EObject _head = IterableExtensions.<EObject>head(ecoreGmRes.getContents());
    final GenModel ecoreGm = ((GenModel) _head);
    GenModel _createGenModel = GenModelFactory.eINSTANCE.createGenModel();
    final Procedure1<GenModel> _function_1 = new Procedure1<GenModel>() {
      @Override
      public void apply(final GenModel it) {
        it.setComplianceLevel(GenJDKLevel.JDK70_LITERAL);
        it.setModelDirectory(ModelTypeExtensions.this.helper.getProject(mt.eResource()).getFolder("src-gen").getFullPath().toString());
        EList<String> _foreignModel = it.getForeignModel();
        _foreignModel.add(ecoreUri);
        it.setModelName(mt.getName());
        it.initialize(pkgs);
        final Consumer<GenPackage> _function = new Consumer<GenPackage>() {
          @Override
          public void accept(final GenPackage gp) {
            gp.setBasePackage(ModelTypeExtensions.this._iQualifiedNameProvider.getFullyQualifiedName(mt).toString().toLowerCase());
          }
        };
        it.getGenPackages().forEach(_function);
        it.getUsedGenPackages().add(IterableExtensions.<GenPackage>head(ecoreGm.getGenPackages()));
      }
    };
    final GenModel genmodel = ObjectExtensions.<GenModel>operator_doubleArrow(_createGenModel, _function_1);
    final Resource res = resSet.createResource(URI.createURI(gmUri));
    EList<EObject> _contents = res.getContents();
    _contents.add(genmodel);
    genmodel.setModelPluginID(this._eclipseProjectHelper.getProject(res).getName());
    try {
      res.save(null);
      return genmodel;
    } catch (final Throwable _t) {
      if (_t instanceof IOException) {
        final IOException e = (IOException)_t;
        ModelTypeExtensions.log.error("Error while serializing new genmodel", e);
        return null;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  /**
   * Creates an in-memory {@link GenModel} for {@code mt} to be used when the
   * serialized version created by {@link #createGenmodel} is not available yet.
   */
  public GenModel createTransientGenmodel(final ModelType mt) {
    final ResourceSetImpl resSet = new ResourceSetImpl();
    resSet.getURIConverter().getURIMap().putAll(EcorePlugin.computePlatformURIMap(true));
    final URI ecoreGmUri = EcorePlugin.getEPackageNsURIToGenModelLocationMap(true).get(EcorePackage.eNS_URI);
    Resource ecoreGmRes = null;
    if ((ecoreGmUri != null)) {
      ecoreGmRes = resSet.getResource(ecoreGmUri, true);
    }
    EList<EObject> _contents = null;
    if (ecoreGmRes!=null) {
      _contents=ecoreGmRes.getContents();
    }
    EObject _head = null;
    if (_contents!=null) {
      _head=IterableExtensions.<EObject>head(_contents);
    }
    final GenModel ecoreGm = ((GenModel) _head);
    GenModel _createGenModel = GenModelFactory.eINSTANCE.createGenModel();
    final Procedure1<GenModel> _function = new Procedure1<GenModel>() {
      @Override
      public void apply(final GenModel it) {
        it.setComplianceLevel(GenJDKLevel.JDK70_LITERAL);
        IProject _project = ModelTypeExtensions.this.helper.getProject(mt.eResource());
        IFolder _folder = null;
        if (_project!=null) {
          _folder=_project.getFolder("src-gen");
        }
        IPath _fullPath = null;
        if (_folder!=null) {
          _fullPath=_folder.getFullPath();
        }
        String _string = null;
        if (_fullPath!=null) {
          _string=_fullPath.toString();
        }
        it.setModelDirectory(_string);
        it.setModelName(mt.getName());
        it.initialize(ModelTypeExtensions.this._modelingElementExtensions.getPkgs(mt));
        final Consumer<GenPackage> _function = new Consumer<GenPackage>() {
          @Override
          public void accept(final GenPackage gp) {
            gp.setBasePackage(ModelTypeExtensions.this._iQualifiedNameProvider.getFullyQualifiedName(mt).toString().toLowerCase());
          }
        };
        it.getGenPackages().forEach(_function);
        if ((ecoreGm != null)) {
          it.getUsedGenPackages().add(IterableExtensions.<GenPackage>head(ecoreGm.getGenPackages()));
        }
      }
    };
    return ObjectExtensions.<GenModel>operator_doubleArrow(_createGenModel, _function);
  }
  
  /**
   * Generates the Java code corresponding to the {@link Genmodel}
   * {@code genModel}. This code generator uses a specialized generation
   * strategy: the code implementing the meta-classes' interfaces is not
   * generated (the .impl package).
   * <br>
   * Only the interfaces and the .util package are generated.
   * 
   * @see Generator#generate
   */
  public void generateModelTypeCode(final GenModel genModel) {
    final String gmUri = GenModelPackage.eNS_URI;
    genModel.reconcile();
    genModel.setCanGenerate(true);
    genModel.setValidateModel(true);
    genModel.setUpdateClasspath(false);
    final GeneratorAdapterFactory.Descriptor.Registry reg = GeneratorAdapterFactory.Descriptor.Registry.INSTANCE;
    final GeneratorAdapterFactory.Descriptor old = IterableExtensions.<GeneratorAdapterFactory.Descriptor>head(reg.getDescriptors(gmUri));
    reg.removeDescriptors(gmUri);
    reg.addDescriptor(gmUri, new GeneratorAdapterFactory.Descriptor() {
      @Override
      public GeneratorAdapterFactory createAdapterFactory() {
        return new ModelTypeGeneratorAdapterFactory();
      }
    });
    Generator _generator = new Generator(reg);
    final Procedure1<Generator> _function = new Procedure1<Generator>() {
      @Override
      public void apply(final Generator it) {
        it.setInput(genModel);
      }
    };
    final Generator generator = ObjectExtensions.<Generator>operator_doubleArrow(_generator, _function);
    generator.generate(genModel, 
      GenBaseGeneratorAdapter.MODEL_PROJECT_TYPE, 
      new Monitor() {
        @Override
        public void beginTask(final String name, final int totalWork) {
        }
        
        @Override
        public void clearBlocked() {
        }
        
        @Override
        public void done() {
        }
        
        @Override
        public void internalWorked(final double work) {
        }
        
        @Override
        public boolean isCanceled() {
          return false;
        }
        
        @Override
        public void setBlocked(final Diagnostic reason) {
        }
        
        @Override
        public void setCanceled(final boolean value) {
        }
        
        @Override
        public void setTaskName(final String name) {
        }
        
        @Override
        public void subTask(final String name) {
        }
        
        @Override
        public void worked(final int work) {
        }
      });
    reg.removeDescriptors(gmUri);
    reg.addDescriptor(gmUri, old);
  }
  
  /**
   * Returns the URI of the {@link ModelType} {@code mt}.
   * May be automatically crafted by configuration or explicitly defined
   * by the user using the 'uri' keywork in the Melange file.
   */
  public String getUri(final ModelType mt) {
    String _xifexpression = null;
    boolean _isExtracted = this.isExtracted(mt);
    if (_isExtracted) {
      _xifexpression = mt.getExtracted().getExactTypeUri();
    } else {
      _xifexpression = mt.getMtUri();
    }
    final String userDefinedUri = _xifexpression;
    String _elvis = null;
    if (userDefinedUri != null) {
      _elvis = userDefinedUri;
    } else {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("http://");
      QualifiedName _lowerCase = this._iQualifiedNameProvider.getFullyQualifiedName(mt).toLowerCase();
      _builder.append(_lowerCase);
      _builder.append("/");
      _elvis = _builder.toString();
    }
    return _elvis;
  }
  
  /**
   * Checks whether the {@link ModelType} {@code mt} is well-formed and can
   * be processed.
   */
  public boolean isValid(final ModelType mt) {
    boolean _and = false;
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(mt.getName());
    boolean _not = (!_isNullOrEmpty);
    if (!_not) {
      _and = false;
    } else {
      boolean _xifexpression = false;
      boolean _isExtracted = this.isExtracted(mt);
      if (_isExtracted) {
        _xifexpression = this._languageExtensions.isValid(mt.getExtracted());
      } else {
        boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<EPackage>filterNull(this._modelingElementExtensions.getPkgs(mt)));
        _xifexpression = (!_isEmpty);
      }
      _and = _xifexpression;
    }
    return _and;
  }
  
  public boolean isComparable(final ModelType mt) {
    boolean _and = false;
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(mt.getName());
    boolean _not = (!_isNullOrEmpty);
    if (!_not) {
      _and = false;
    } else {
      boolean _xifexpression = false;
      boolean _isExtracted = this.isExtracted(mt);
      if (_isExtracted) {
        _xifexpression = this._languageExtensions.isTypable(mt.getExtracted());
      } else {
        boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<EPackage>filterNull(this._modelingElementExtensions.getPkgs(mt)));
        _xifexpression = (!_isEmpty);
      }
      _and = _xifexpression;
    }
    return _and;
  }
  
  /**
   * Returns whether the {@link ModelType} {@code mt} is extracted from
   * a {@link Language} implementation or manually crafted.
   */
  public boolean isExtracted(final ModelType mt) {
    Language _extracted = mt.getExtracted();
    return (_extracted != null);
  }
  
  /**
   * Return true if the {@link ModelType} {@code mt} is extracted from
   * an external {@link Language}
   */
  public boolean isExternal(final ModelType mt) {
    return (this.isExtracted(mt) && (mt.getExtracted() instanceof ExternalLanguage));
  }
  
  /**
   * Checks whether the {@link ModelType} {@code mt1} is a subtype of the
   * {@link ModelType} {@code mt2}.
   * 
   * @see MatchingHelper#match
   */
  public boolean isSubtypeOf(final ModelType mt1, final ModelType mt2) {
    return this.matchingHelper.match(
      IterableExtensions.<EPackage>toList(this._modelingElementExtensions.getPkgs(mt1)), IterableExtensions.<EPackage>toList(this._modelingElementExtensions.getPkgs(mt2)), null);
  }
}
