/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.eclipse;

import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import com.google.inject.Inject;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.metamodel.melange.Language;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.function.Consumer;
import java.util.jar.Attributes;
import org.apache.log4j.Logger;
import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gemoc.commons.eclipse.pde.classpath.BuildPropertiesHelper;
import org.eclipse.gemoc.commons.eclipse.pde.classpath.ClasspathHelper;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.pde.internal.core.natures.PDE;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.util.MergeableManifest;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * A collection of utilities around Eclipse's APIs to manage the creation,
 * maintenance, etc. of the Eclipse projects/MANIFEST/plugin.xml generated
 * by Melange.
 */
@SuppressWarnings("all")
public class EclipseProjectHelper {
  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;
  
  private final static Logger log = Logger.getLogger(EclipseProjectHelper.class);
  
  public final static String GEMOCNatureID = "org.eclipse.gemoc.execution.sequential.javaxdsml.ide.ui.GemocSequentialLanguageNature";
  
  /**
   * Returns the {@link IProject} containing the Melange file pointed by
   * the {@link Resource} {@code res}, or null if in a standalone context.
   */
  public IProject getProject(final Resource res) {
    try {
      IProject _xifexpression = null;
      if (((res != null) && (res.getURI().toPlatformString(true) != null))) {
        IWorkspaceRoot _root = ResourcesPlugin.getWorkspace().getRoot();
        String _platformString = res.getURI().toPlatformString(true);
        Path _path = new Path(_platformString);
        _xifexpression = _root.getFile(_path).getProject();
      } else {
        _xifexpression = null;
      }
      return _xifexpression;
    } catch (final Throwable _t) {
      if (_t instanceof IllegalStateException) {
        final IllegalStateException e = (IllegalStateException)_t;
        EclipseProjectHelper.log.info("Couldn\'t access platform", e);
        return null;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  /**
   * Returns the qualifiers of the bundles the {@link IProject} {@code project}
   * depends on, extracted from its MANIFEST.MF.
   */
  public Iterable<String> getDependencies(final IProject project) {
    final IFile manifestFile = project.getFile("META-INF/MANIFEST.MF");
    if ((((manifestFile != null) && manifestFile.exists()) && manifestFile.isAccessible())) {
      InputStream input = null;
      try {
        input = manifestFile.getContents();
        final MergeableManifest manifest = new MergeableManifest(input);
        final Attributes attrs = manifest.getMainAttributes();
        final Iterable<String> bundles = Splitter.on(",").omitEmptyStrings().trimResults().split(attrs.getValue("Require-Bundle"));
        final Function1<String, String> _function = new Function1<String, String>() {
          @Override
          public String apply(final String it) {
            return IterableExtensions.<String>head(((Iterable<String>)Conversions.doWrapArray(it.split(";"))));
          }
        };
        return IterableExtensions.<String, String>map(bundles, _function);
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception e = (Exception)_t;
          EclipseProjectHelper.log.error("Couldn\'t retrieve MANIFEST.MF dependencies", e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      } finally {
        this.closeQuietly(input);
      }
      return CollectionLiterals.<String>newArrayList();
    }
    return null;
  }
  
  /**
   * Updates the MANIFEST.MF of the {@link IProject} {@code project}
   * with the given list of {@code bundles}.
   */
  public void addDependencies(final IProject project, final Iterable<String> bundles) {
    final IFile manifestFile = project.getFile("META-INF/MANIFEST.MF");
    if (((((manifestFile != null) && manifestFile.exists()) && manifestFile.isAccessible()) && (!manifestFile.getResourceAttributes().isReadOnly()))) {
      OutputStream output = null;
      InputStream input = null;
      try {
        boolean _isSynchronized = manifestFile.isSynchronized(IResource.DEPTH_ZERO);
        boolean _not = (!_isSynchronized);
        if (_not) {
          manifestFile.refreshLocal(IResource.DEPTH_ZERO, null);
        }
        input = manifestFile.getContents();
        final MergeableManifest manifest = new MergeableManifest(input);
        manifest.addRequiredBundles(Sets.<String>newHashSet(bundles));
        final ByteArrayOutputStream out = new ByteArrayOutputStream();
        BufferedOutputStream _bufferedOutputStream = new BufferedOutputStream(out);
        output = _bufferedOutputStream;
        manifest.write(output);
        byte[] _byteArray = out.toByteArray();
        final ByteArrayInputStream in = new ByteArrayInputStream(_byteArray);
        BufferedInputStream _bufferedInputStream = new BufferedInputStream(in);
        input = _bufferedInputStream;
        manifestFile.setContents(input, true, true, null);
        final Consumer<String> _function = new Consumer<String>() {
          @Override
          public void accept(final String it) {
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("Dependendency ");
            _builder.append(it);
            _builder.append(" added to ");
            _builder.append(project);
            EclipseProjectHelper.log.debug(_builder);
          }
        };
        bundles.forEach(_function);
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception e = (Exception)_t;
          EclipseProjectHelper.log.error("Couldn\'t update MANIFEST.MF", e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      } finally {
        this.closeQuietly(input);
        this.closeQuietly(output);
      }
    }
  }
  
  /**
   * Updates the MANIFEST.MF of the {@link IProject} {@code project}
   * with the given list of {@code bundles}.
   */
  public void addExportedPackages(final IProject project, final Iterable<String> packages) {
    final IFile manifestFile = project.getFile("META-INF/MANIFEST.MF");
    if (((((manifestFile != null) && manifestFile.exists()) && manifestFile.isAccessible()) && (!manifestFile.getResourceAttributes().isReadOnly()))) {
      OutputStream output = null;
      InputStream input = null;
      try {
        boolean _isSynchronized = manifestFile.isSynchronized(IResource.DEPTH_ZERO);
        boolean _not = (!_isSynchronized);
        if (_not) {
          manifestFile.refreshLocal(IResource.DEPTH_ZERO, null);
        }
        input = manifestFile.getContents();
        final MergeableManifest manifest = new MergeableManifest(input);
        manifest.addExportedPackages(Sets.<String>newHashSet(packages));
        final ByteArrayOutputStream out = new ByteArrayOutputStream();
        BufferedOutputStream _bufferedOutputStream = new BufferedOutputStream(out);
        output = _bufferedOutputStream;
        manifest.write(output);
        byte[] _byteArray = out.toByteArray();
        final ByteArrayInputStream in = new ByteArrayInputStream(_byteArray);
        BufferedInputStream _bufferedInputStream = new BufferedInputStream(in);
        input = _bufferedInputStream;
        manifestFile.setContents(input, true, true, null);
        final Consumer<String> _function = new Consumer<String>() {
          @Override
          public void accept(final String it) {
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("Exported package ");
            _builder.append(it);
            _builder.append(" added to ");
            _builder.append(project);
            EclipseProjectHelper.log.debug(_builder);
          }
        };
        packages.forEach(_function);
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception e = (Exception)_t;
          EclipseProjectHelper.log.error("Couldn\'t update MANIFEST.MF", e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      } finally {
        this.closeQuietly(input);
        this.closeQuietly(output);
      }
    }
  }
  
  private void closeQuietly(final Closeable c) {
    if ((c == null)) {
      return;
    }
    try {
      c.close();
    } catch (final Throwable _t) {
      if (_t instanceof IOException) {
        final IOException e = (IOException)_t;
        EclipseProjectHelper.log.error("Couldn\'t close resource", e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  /**
   * Removes the set of {@code bundles} from the MANIFEST.MF of the
   * {@link IProject} {@code project}.
   */
  public void removeDependencies(final IProject project, final Iterable<String> bundles) {
    boolean _isEmpty = IterableExtensions.isEmpty(bundles);
    if (_isEmpty) {
      return;
    }
    final IFile manifestFile = project.getFile("META-INF/MANIFEST.MF");
    if (((((manifestFile != null) && manifestFile.exists()) && manifestFile.isAccessible()) && (!manifestFile.getResourceAttributes().isReadOnly()))) {
      OutputStream output = null;
      InputStream input = null;
      try {
        boolean _isSynchronized = manifestFile.isSynchronized(IResource.DEPTH_ZERO);
        boolean _not = (!_isSynchronized);
        if (_not) {
          manifestFile.refreshLocal(IResource.DEPTH_ZERO, null);
        }
        input = manifestFile.getContents();
        final MergeableManifest manifest = new MergeableManifest(input);
        Object _get = manifest.getMainAttributes().get(MergeableManifest.REQUIRE_BUNDLE);
        String requiredBundles = ((String) _get);
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("(?m)^ ?(");
        String _join = IterableExtensions.join(bundles, "|");
        _builder.append(_join);
        _builder.append(").*$(?:\\r?\\n)?");
        final String regex = _builder.toString();
        final String result = requiredBundles.replaceAll(regex, "");
        manifest.getMainAttributes().put(MergeableManifest.REQUIRE_BUNDLE, result.replaceFirst(",$", ""));
        final ByteArrayOutputStream out = new ByteArrayOutputStream();
        BufferedOutputStream _bufferedOutputStream = new BufferedOutputStream(out);
        output = _bufferedOutputStream;
        manifest.write(output);
        byte[] _byteArray = out.toByteArray();
        final ByteArrayInputStream in = new ByteArrayInputStream(_byteArray);
        BufferedInputStream _bufferedInputStream = new BufferedInputStream(in);
        input = _bufferedInputStream;
        manifestFile.setContents(input, true, true, null);
        final Consumer<String> _function = new Consumer<String>() {
          @Override
          public void accept(final String it) {
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("Dependendency ");
            _builder.append(it);
            _builder.append(" removed from ");
            _builder.append(project);
            EclipseProjectHelper.log.debug(_builder);
          }
        };
        bundles.forEach(_function);
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception e = (Exception)_t;
          EclipseProjectHelper.log.error("Couldn\'t update MANIFEST.MF dependencies", e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      } finally {
        this.closeQuietly(input);
        this.closeQuietly(output);
      }
    }
  }
  
  /**
   * add the {@link Language} {@code l} in the given Melange project.
   * It will update it instead of creating a new one as does {@link #createEMFRuntimeProject}
   * 
   * In order to be able to clean the project, the ecore is generated in model-gen
   * and the java code for the ecore is generated in src-model-gen
   * 
   * Additions to the usual Melange:
   * <ul>
   *   <li>Source folders: src-model-gen</li>
   *   <li>Nature: GEMOC Nature</li>
   * </ul>
   */
  public void createEMFRuntimeInMelangeProject(final IProject project, final Language l, final IProgressMonitor monitor) {
    try {
      final SubMonitor subMonitor = SubMonitor.convert(monitor, 100);
      final IFolder srcModelGenFolder = project.getFolder("src-model-gen");
      boolean _exists = srcModelGenFolder.exists();
      if (_exists) {
        srcModelGenFolder.delete(true, true, subMonitor.split(10));
      }
      srcModelGenFolder.create(false, true, subMonitor.split(50));
      ClasspathHelper.addSourceEntry(project, "src-model-gen", subMonitor.split(10));
      BuildPropertiesHelper.addMainJarSourceEntry(project, "src-model-gen", subMonitor.split(10));
      if (((ResourcesPlugin.getWorkspace().getNatureDescriptor(EclipseProjectHelper.GEMOCNatureID) != null) && 
        (!project.hasNature(EclipseProjectHelper.GEMOCNatureID)))) {
        final IProjectDescription description = project.getDescription();
        final String[] natures = description.getNatureIds();
        int _length = natures.length;
        int _plus = (_length + 1);
        final String[] newNatures = new String[_plus];
        System.arraycopy(natures, 0, newNatures, 1, natures.length);
        newNatures[0] = EclipseProjectHelper.GEMOCNatureID;
        description.setNatureIds(newNatures);
        project.setDescription(description, null);
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Creates a new Eclipse project named {@code projectName} for the
   * {@link Language} {@code l}. Acting as a GEMOC Language
   * <ul>
   *   <li>Natures: GEMOCSequentialLanguage, JAVA, PLUGIN, XText</li>
   *   <li>Builders: GEMOCSequentialBuilder, JAVA, MANIFEST, SCHEMA</li>
   *   <li>Source folders: src/src-gen</li>
   *   <li>Dependencies: Ecore, K3, Xbase</li>
   * </ul>
   */
  public IProject createGemocLangEMFRuntimeProject(final String projectName, final Language l) {
    try {
      List<String> _xifexpression = null;
      boolean _hasCopiedAspects = this._languageExtensions.hasCopiedAspects(l);
      if (_hasCopiedAspects) {
        String _aspectsNamespace = this._languageExtensions.getAspectsNamespace(l);
        _xifexpression = Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_aspectsNamespace));
      } else {
        _xifexpression = Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList());
      }
      NullProgressMonitor _nullProgressMonitor = new NullProgressMonitor();
      final IProject project = this.createEclipseProject(projectName, 
        Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(EclipseProjectHelper.GEMOCNatureID, JavaCore.NATURE_ID, PDE.PLUGIN_NATURE, "org.eclipse.xtext.ui.shared.xtextNature")), 
        Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(EclipseProjectHelper.GEMOCNatureID, JavaCore.BUILDER_ID, PDE.MANIFEST_BUILDER_ID, PDE.SCHEMA_BUILDER_ID)), 
        Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("src", "src-gen")), 
        Collections.<IProject>unmodifiableList(CollectionLiterals.<IProject>newArrayList()), 
        Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("org.eclipse.emf.ecore", "fr.inria.diverse.k3.al.annotationprocessor.plugin", "fr.inria.diverse.melange", "org.eclipse.xtext.xbase.lib")), _xifexpression, 
        Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList()), _nullProgressMonitor);
      final IFolder modelFolder = project.getFolder("model");
      modelFolder.create(false, true, null);
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Runtime EMF project ");
      _builder.append(project);
      _builder.append(" created.");
      EclipseProjectHelper.log.debug(_builder);
      return project;
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        EclipseProjectHelper.log.error("Unexpected exception while creating new runtime project", e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return null;
  }
  
  /**
   * Creates a new Eclipse project named {@code projectName} for the
   * {@link Language} {@code l}.
   * <ul>
   *   <li>Natures: JAVA, PLUGIN, XText</li>
   *   <li>Builders: JAVA, MANIFEST, SCHEMA</li>
   *   <li>Source folders: src/src-gen</li>
   *   <li>Dependencies: Ecore, K3, Xbase</li>
   * </ul>
   */
  public IProject createEMFRuntimeProject(final String projectName, final Language l) {
    try {
      List<String> _xifexpression = null;
      boolean _hasCopiedAspects = this._languageExtensions.hasCopiedAspects(l);
      if (_hasCopiedAspects) {
        String _aspectsNamespace = this._languageExtensions.getAspectsNamespace(l);
        _xifexpression = Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_aspectsNamespace));
      } else {
        _xifexpression = Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList());
      }
      NullProgressMonitor _nullProgressMonitor = new NullProgressMonitor();
      final IProject project = this.createEclipseProject(projectName, 
        Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(JavaCore.NATURE_ID, PDE.PLUGIN_NATURE, "org.eclipse.xtext.ui.shared.xtextNature")), 
        Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(JavaCore.BUILDER_ID, PDE.MANIFEST_BUILDER_ID, PDE.SCHEMA_BUILDER_ID)), 
        Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("src", "src-gen")), 
        Collections.<IProject>unmodifiableList(CollectionLiterals.<IProject>newArrayList()), 
        Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("org.eclipse.emf.ecore", "fr.inria.diverse.k3.al.annotationprocessor.plugin", "fr.inria.diverse.melange", "org.eclipse.xtext.xbase.lib")), _xifexpression, 
        Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList()), _nullProgressMonitor);
      final IFolder modelFolder = project.getFolder("model");
      modelFolder.create(false, true, null);
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Runtime EMF project ");
      _builder.append(project);
      _builder.append(" created.");
      EclipseProjectHelper.log.debug(_builder);
      return project;
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        EclipseProjectHelper.log.error("Unexpected exception while creating new runtime project", e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return null;
  }
  
  /**
   * Wololo, wololo wololo.
   */
  private IProject createEclipseProject(final String name, final Iterable<String> natures, final Iterable<String> builders, final Iterable<String> srcFolders, final Iterable<IProject> referencedProjects, final Iterable<String> requiredBundles, final Iterable<String> exportedPackages, final Iterable<String> extensions, final IProgressMonitor monitor) {
    try {
      monitor.beginTask("", 10);
      monitor.subTask(("Creating project " + name));
      final IWorkspace workspace = ResourcesPlugin.getWorkspace();
      final IProject project = workspace.getRoot().getProject(name);
      IPath previousProjectLocation = null;
      boolean _exists = project.exists();
      if (_exists) {
        previousProjectLocation = project.getLocation();
        SubProgressMonitor _subProgressMonitor = new SubProgressMonitor(monitor, 1);
        project.delete(true, true, _subProgressMonitor);
      }
      final IJavaProject javaProject = JavaCore.create(project);
      final IProjectDescription description = workspace.newProjectDescription(name);
      description.setLocation(previousProjectLocation);
      SubProgressMonitor _subProgressMonitor_1 = new SubProgressMonitor(monitor, 1);
      project.create(description, _subProgressMonitor_1);
      final ArrayList<IClasspathEntry> classpathEntries = CollectionLiterals.<IClasspathEntry>newArrayList();
      boolean _isEmpty = IterableExtensions.isEmpty(referencedProjects);
      boolean _not = (!_isEmpty);
      if (_not) {
        description.setReferencedProjects(((IProject[])Conversions.unwrapArray(referencedProjects, IProject.class)));
        final Function1<IProject, IClasspathEntry> _function = new Function1<IProject, IClasspathEntry>() {
          @Override
          public IClasspathEntry apply(final IProject it) {
            return JavaCore.newProjectEntry(it.getFullPath());
          }
        };
        Iterable<IClasspathEntry> _map = IterableExtensions.<IProject, IClasspathEntry>map(referencedProjects, _function);
        Iterables.<IClasspathEntry>addAll(classpathEntries, _map);
      }
      description.setNatureIds(((String[])Conversions.unwrapArray(natures, String.class)));
      final Function1<String, ICommand> _function_1 = new Function1<String, ICommand>() {
        @Override
        public ICommand apply(final String buildName) {
          ICommand _newCommand = description.newCommand();
          final Procedure1<ICommand> _function = new Procedure1<ICommand>() {
            @Override
            public void apply(final ICommand it) {
              it.setBuilderName(buildName);
            }
          };
          return ObjectExtensions.<ICommand>operator_doubleArrow(_newCommand, _function);
        }
      };
      description.setBuildSpec(((ICommand[])Conversions.unwrapArray(IterableExtensions.<String, ICommand>map(builders, _function_1), ICommand.class)));
      SubProgressMonitor _subProgressMonitor_2 = new SubProgressMonitor(monitor, 1);
      project.open(_subProgressMonitor_2);
      SubProgressMonitor _subProgressMonitor_3 = new SubProgressMonitor(monitor, 1);
      project.setDescription(description, _subProgressMonitor_3);
      final Consumer<String> _function_2 = new Consumer<String>() {
        @Override
        public void accept(final String src) {
          final IFolder container = project.getFolder(src);
          try {
            boolean _exists = container.exists();
            boolean _not = (!_exists);
            if (_not) {
              SubProgressMonitor _subProgressMonitor = new SubProgressMonitor(monitor, 1);
              container.create(false, true, _subProgressMonitor);
            }
            classpathEntries.add(0, JavaCore.newSourceEntry(container.getFullPath()));
          } catch (final Throwable _t) {
            if (_t instanceof CoreException) {
              final CoreException e = (CoreException)_t;
              EclipseProjectHelper.log.error("Couldn\'t update project classpath", e);
            } else {
              throw Exceptions.sneakyThrow(_t);
            }
          }
        }
      };
      srcFolders.forEach(_function_2);
      Path _path = new Path("org.eclipse.jdt.launching.JRE_CONTAINER");
      IClasspathEntry _newContainerEntry = JavaCore.newContainerEntry(_path);
      classpathEntries.add(_newContainerEntry);
      Path _path_1 = new Path("org.eclipse.pde.core.requiredPlugins");
      IClasspathEntry _newContainerEntry_1 = JavaCore.newContainerEntry(_path_1);
      classpathEntries.add(_newContainerEntry_1);
      final IFolder binFolder = project.getFolder("bin");
      SubProgressMonitor _subProgressMonitor_4 = new SubProgressMonitor(monitor, 1);
      binFolder.create(false, true, _subProgressMonitor_4);
      SubProgressMonitor _subProgressMonitor_5 = new SubProgressMonitor(monitor, 1);
      javaProject.setRawClasspath(((IClasspathEntry[])Conversions.unwrapArray(classpathEntries, IClasspathEntry.class)), _subProgressMonitor_5);
      IPath _fullPath = binFolder.getFullPath();
      SubProgressMonitor _subProgressMonitor_6 = new SubProgressMonitor(monitor, 1);
      javaProject.setOutputLocation(_fullPath, _subProgressMonitor_6);
      this.createManifest(name, requiredBundles, exportedPackages, monitor, project);
      this.createPluginXml(project, extensions, monitor);
      this.createBuildProperties(project, srcFolders, monitor);
      return project;
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        EclipseProjectHelper.log.error("Unexpected exception while generating new project", e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return null;
  }
  
  private void createManifest(final String name, final Iterable<String> requiredBundles, final Iterable<String> exportedPackages, final IProgressMonitor monitor, final IProject project) throws CoreException {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Manifest-Version: 1.0");
    _builder.newLine();
    _builder.append("Bundle-ManifestVersion: 2");
    _builder.newLine();
    _builder.append("Bundle-Name: ");
    _builder.append(name);
    _builder.newLineIfNotEmpty();
    _builder.append("Bundle-SymbolicName: ");
    _builder.append(name);
    _builder.append(";singleton:=true");
    _builder.newLineIfNotEmpty();
    _builder.append("Bundle-Version: 0.1.0");
    _builder.newLine();
    {
      boolean _isEmpty = IterableExtensions.isEmpty(requiredBundles);
      boolean _not = (!_isEmpty);
      if (_not) {
        _builder.append("Require-Bundle: ");
        {
          boolean _hasElements = false;
          for(final String b : requiredBundles) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(",\n  ", "");
            }
            _builder.append(b);
          }
        }
        _builder.newLineIfNotEmpty();
      }
    }
    {
      boolean _isEmpty_1 = IterableExtensions.isEmpty(exportedPackages);
      boolean _not_1 = (!_isEmpty_1);
      if (_not_1) {
        _builder.append("Export-Package: ");
        {
          boolean _hasElements_1 = false;
          for(final String p : exportedPackages) {
            if (!_hasElements_1) {
              _hasElements_1 = true;
            } else {
              _builder.appendImmediate(",\n  ", "");
            }
            _builder.append(p);
          }
        }
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("Bundle-RequiredExecutionEnvironment: JavaSE-1.7");
    _builder.newLine();
    _builder.newLine();
    final String content = _builder.toString();
    final IFolder metaInf = project.getFolder("META-INF");
    SubProgressMonitor _subProgressMonitor = new SubProgressMonitor(monitor, 1);
    metaInf.create(false, true, _subProgressMonitor);
    this.createFile("MANIFEST.MF", metaInf, content, monitor);
  }
  
  private void createPluginXml(final IProject project, final Iterable<String> extensions, final IProgressMonitor monitor) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
    _builder.newLine();
    _builder.append("<?eclipse version=\"3.0\"?>");
    _builder.newLine();
    _builder.newLine();
    _builder.append("<plugin>");
    _builder.newLine();
    _builder.newLine();
    {
      for(final String e : extensions) {
        _builder.append(e);
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    _builder.append("</plugin>");
    _builder.newLine();
    final String content = _builder.toString();
    this.createFile("plugin.xml", project, content, monitor);
  }
  
  private void createBuildProperties(final IProject project, final Iterable<String> srcFolders, final IProgressMonitor monitor) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("source.. = ");
    {
      boolean _hasElements = false;
      for(final String f : srcFolders) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(",\\n  ", "");
        }
        _builder.append(f);
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("bin.includes = META-INF/,\\");
    _builder.newLine();
    _builder.append("  ");
    _builder.append(".");
    _builder.newLine();
    final String content = _builder.toString();
    this.createFile("build.properties", project, content, monitor);
  }
  
  private IFile createFile(final String name, final IContainer container, final String content, final IProgressMonitor monitor) {
    Path _path = new Path(name);
    final IFile f = container.getFile(_path);
    InputStream stream = null;
    try {
      byte[] _bytes = content.getBytes(f.getCharset());
      ByteArrayInputStream _byteArrayInputStream = new ByteArrayInputStream(_bytes);
      stream = _byteArrayInputStream;
      boolean _exists = f.exists();
      if (_exists) {
        f.setContents(stream, true, true, monitor);
      } else {
        f.create(stream, true, monitor);
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        EclipseProjectHelper.log.error("Error while creating new IFile", e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    } finally {
      this.closeQuietly(stream);
    }
    monitor.worked(1);
    return f;
  }
  
  /**
   * @deprecated Use {@link #createEMFRuntimeProject} instead.
   */
  public IProject createAspectsRuntimeProject(final IProject original, final String projectName, final String generatedPackage, final String emfRuntimeBundle) {
    final HashSet<String> dependencies = Sets.<String>newHashSet(this.getDependencies(original));
    dependencies.add("org.eclipse.emf.ecore");
    dependencies.add("fr.inria.diverse.k3.al.annotationprocessor.plugin");
    dependencies.add(emfRuntimeBundle);
    NullProgressMonitor _nullProgressMonitor = new NullProgressMonitor();
    final IProject project = this.createEclipseProject(projectName, 
      Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(JavaCore.NATURE_ID, PDE.PLUGIN_NATURE)), 
      Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(JavaCore.BUILDER_ID, PDE.MANIFEST_BUILDER_ID, PDE.SCHEMA_BUILDER_ID)), 
      Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("src-gen")), 
      Collections.<IProject>unmodifiableList(CollectionLiterals.<IProject>newArrayList()), dependencies, 
      Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(generatedPackage)), 
      Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList()), _nullProgressMonitor);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Runtime aspects project ");
    _builder.append(project);
    _builder.append(" created.");
    EclipseProjectHelper.log.debug(_builder);
    return project;
  }
  
  public void addNature(final IProject project, final String natureID) {
    try {
      final IProjectDescription desc = project.getDescription();
      final String[] natures = desc.getNatureIds();
      final ArrayList<String> newNatures = CollectionLiterals.<String>newArrayList(natureID);
      CollectionExtensions.<String>addAll(newNatures, natures);
      desc.setNatureIds(((String[])Conversions.unwrapArray(newNatures, String.class)));
      NullProgressMonitor _nullProgressMonitor = new NullProgressMonitor();
      project.setDescription(desc, _nullProgressMonitor);
    } catch (final Throwable _t) {
      if (_t instanceof CoreException) {
        final CoreException e = (CoreException)_t;
        e.printStackTrace();
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  public void addBuilder(final IProject project, final String builderID) {
    try {
      final IProjectDescription projectDescription = project.getDescription();
      final ICommand[] buildSpec = projectDescription.getBuildSpec();
      final ICommand command = projectDescription.newCommand();
      command.setBuilderName(builderID);
      final ArrayList<ICommand> newBuildSpect = CollectionLiterals.<ICommand>newArrayList(command);
      CollectionExtensions.<ICommand>addAll(newBuildSpect, buildSpec);
      projectDescription.setBuildSpec(((ICommand[])Conversions.unwrapArray(newBuildSpect, ICommand.class)));
      NullProgressMonitor _nullProgressMonitor = new NullProgressMonitor();
      project.setDescription(projectDescription, _nullProgressMonitor);
    } catch (final Throwable _t) {
      if (_t instanceof CoreException) {
        final CoreException e = (CoreException)_t;
        e.printStackTrace();
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
}
