/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.utils;

import com.google.common.base.Objects;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.SetMultimap;
import fr.inria.diverse.melange.ast.AspectExtensions;
import fr.inria.diverse.melange.metamodel.melange.Aspect;
import fr.inria.diverse.melange.metamodel.melange.ClassBinding;
import fr.inria.diverse.melange.metamodel.melange.PackageBinding;
import fr.inria.diverse.melange.metamodel.melange.PropertyBinding;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

@SuppressWarnings("all")
public class RenamingRuleManager {
  @Extension
  private AspectExtensions aspectExtension;

  private List<PackageBinding> sourceBinding;

  private final List<Pair<String, String>> classRules = CollectionLiterals.<Pair<String, String>>newArrayList();

  private final List<Pair<String, String>> packageRules = CollectionLiterals.<Pair<String, String>>newArrayList();

  private final List<Pair<String, String>> propertiesRules = CollectionLiterals.<Pair<String, String>>newArrayList();

  private final SetMultimap<String, Pair<String, String>> propertiesAspectRules = HashMultimap.<String, Pair<String, String>>create();

  public RenamingRuleManager(final List<PackageBinding> renamingRules, final List<Aspect> aspects, final AspectExtensions aspectExtension) {
    this.aspectExtension = aspectExtension;
    this.storeRenamingRules(renamingRules);
    this.storeRenamedAspectProperties(aspects);
  }

  /**
   * Store renaming rules as pairs of String and group them by: <br>
   * - Packages renaming <br>
   * - Class renaming <br>
   * - Property renaming
   */
  public void storeRenamingRules(final List<PackageBinding> renamingRules) {
    if ((renamingRules != null)) {
      this.sourceBinding = renamingRules;
      final Consumer<PackageBinding> _function = (PackageBinding packRule) -> {
        String _from = packRule.getFrom();
        String _to = packRule.getTo();
        Pair<String, String> _mappedTo = Pair.<String, String>of(_from, _to);
        this.packageRules.add(_mappedTo);
        final Consumer<ClassBinding> _function_1 = (ClassBinding classRule) -> {
          String _from_1 = packRule.getFrom();
          String _plus = (_from_1 + ".");
          String _from_2 = classRule.getFrom();
          String _plus_1 = (_plus + _from_2);
          String _to_1 = packRule.getTo();
          String _plus_2 = (_to_1 + ".");
          String _to_2 = classRule.getTo();
          String _plus_3 = (_plus_2 + _to_2);
          Pair<String, String> _mappedTo_1 = Pair.<String, String>of(_plus_1, _plus_3);
          this.classRules.add(_mappedTo_1);
          final Consumer<PropertyBinding> _function_2 = (PropertyBinding propRule) -> {
            String _from_3 = packRule.getFrom();
            String _plus_4 = (_from_3 + ".");
            String _from_4 = classRule.getFrom();
            String _plus_5 = (_plus_4 + _from_4);
            String _plus_6 = (_plus_5 + ".");
            String _from_5 = propRule.getFrom();
            String _plus_7 = (_plus_6 + _from_5);
            String _to_3 = packRule.getTo();
            String _plus_8 = (_to_3 + ".");
            String _to_4 = classRule.getTo();
            String _plus_9 = (_plus_8 + _to_4);
            String _plus_10 = (_plus_9 + ".");
            String _to_5 = propRule.getTo();
            String _plus_11 = (_plus_10 + _to_5);
            Pair<String, String> _mappedTo_2 = Pair.<String, String>of(_plus_7, _plus_11);
            this.propertiesRules.add(_mappedTo_2);
          };
          classRule.getProperties().forEach(_function_2);
        };
        packRule.getClasses().forEach(_function_1);
      };
      renamingRules.forEach(_function);
    }
  }

  /**
   * Search properties defined in Aspect and find matching renaming rules.
   * Store the selected rules and group them by Aspect.
   * 
   * @return AspectName -> (package.sourceClass.property, package.targetClass.property)
   */
  public void storeRenamedAspectProperties(final List<Aspect> aspects) {
    final Function1<Aspect, Boolean> _function = (Aspect it) -> {
      return Boolean.valueOf(this.aspectExtension.hasAspectAnnotation(it));
    };
    final Consumer<Aspect> _function_1 = (Aspect asp) -> {
      final String targetClass = this.aspectExtension.getTargetedClassFqn(asp);
      JvmType _type = asp.getAspectTypeRef().getType();
      final JvmGenericType type = ((JvmGenericType) _type);
      final Consumer<JvmOperation> _function_2 = (JvmOperation op) -> {
        final String name = op.getSimpleName();
        final Function1<Pair<String, String>, Boolean> _function_3 = (Pair<String, String> rule) -> {
          String _key = rule.getKey();
          return Boolean.valueOf(Objects.equal(_key, ((targetClass + ".") + name)));
        };
        final Pair<String, String> rule = IterableExtensions.<Pair<String, String>>findFirst(this.propertiesRules, _function_3);
        if ((rule != null)) {
          this.propertiesAspectRules.put(type.getSimpleName(), rule);
        }
      };
      Iterables.<JvmOperation>filter(type.getMembers(), JvmOperation.class).forEach(_function_2);
    };
    IterableExtensions.<Aspect>filter(aspects, _function).forEach(_function_1);
  }

  /**
   * Return a renaming rule for the package {@link packageName}.
   * Return null if none.<br>
   * 
   * @return sourcePackage -> targetPackage
   */
  public Pair<String, String> getPackageRule(final String packageName) {
    final Function1<Pair<String, String>, Boolean> _function = (Pair<String, String> it) -> {
      String _key = it.getKey();
      return Boolean.valueOf(Objects.equal(_key, packageName));
    };
    return IterableExtensions.<Pair<String, String>>findFirst(this.packageRules, _function);
  }

  /**
   * Return a renaming rule for the class {@link qualifiedClassName}.
   * Return null if none.<br>
   * 
   * @return sourcePackage.sourceClass -> targetPackage.targetClass
   */
  public Pair<String, String> getClassRule(final String qualifiedClassName) {
    final Function1<Pair<String, String>, Boolean> _function = (Pair<String, String> it) -> {
      String _key = it.getKey();
      return Boolean.valueOf(Objects.equal(_key, qualifiedClassName));
    };
    return IterableExtensions.<Pair<String, String>>findFirst(this.classRules, _function);
  }

  /**
   * Return a renaming rule for the property {@link qualifiedPropertyName}.
   * Return null if none.<br>
   * 
   * @return sourcePackage.sourceClass.sourceProperty -> targetPackage.targetClass.targetProperty
   */
  public Pair<String, String> getPropertyRule(final String qualifiedPropertyName) {
    final Function1<Pair<String, String>, Boolean> _function = (Pair<String, String> it) -> {
      String _key = it.getKey();
      return Boolean.valueOf(Objects.equal(_key, qualifiedPropertyName));
    };
    return IterableExtensions.<Pair<String, String>>findFirst(this.propertiesRules, _function);
  }

  /**
   * Return renaming rules for properties defined in {@link aspectName}.
   * Return null if none.
   */
  public Set<Pair<String, String>> getRulesForAspect(final String aspectName) {
    return this.propertiesAspectRules.get(aspectName);
  }

  public List<Pair<String, String>> getAllPackageRules() {
    return this.packageRules;
  }

  public List<Pair<String, String>> getAllClassRules() {
    return this.classRules;
  }

  public List<Pair<String, String>> getAllPropertyRules() {
    return this.propertiesRules;
  }
}
