/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.tools.actionscript.debug;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.debug.core.IJavaLineBreakpoint;

import flash.tools.debugger.Location;
import flash.tools.debugger.SourceFile;

public class ActionScriptLineMapper {

	public static class FileLocation {
		int lineNumber;
		String fileName;
		String qualifiedName;
	}
	
	static ActionScriptLineMapper mapper;
	
	public static synchronized ActionScriptLineMapper getInstance () {
		if (mapper == null) {
			mapper = new ActionScriptLineMapper ();
		}
		return mapper;
	}
	
	public FileLocation getActionScriptLocation (IJavaLineBreakpoint breakpoint) throws CoreException {
		String typeName = breakpoint.getTypeName();
		typeName = typeName.replace('.', File.separatorChar);
		IResource resource = breakpoint.getMarker().getResource();
		typeName = resource.getProject().getLocation().toOSString() + File.separator + ".buildas" + File.separator + typeName;
		String searchString = typeName.substring(0, typeName.lastIndexOf(File.separatorChar));
		File directory = new File(searchString);
		File[] searchResults = directory.listFiles();
		String javaName = resource.getLocation().toOSString();
		for (int i = 0; i < searchResults.length; i++) {
			String currentFilePath = searchResults[i].getAbsolutePath();
			if (currentFilePath.startsWith(typeName) && currentFilePath.endsWith(".map")) {	
				String localTypeName = currentFilePath.substring(0, currentFilePath.lastIndexOf('.'));
				FileLocation location = findLocation(localTypeName, javaName, breakpoint.getLineNumber());
				if (location != null) return location;
			}
		}
		return null;
	}
	
	public FileLocation getJavaLocation (Location location) throws CoreException {
		SourceFile sourceFile = location.getFile();
		if (sourceFile == null) return null;
		String filePath = sourceFile.getFullPath();
		int lookupLine = location.getLine();
		int dotIndex = filePath.lastIndexOf('.');
		filePath = filePath.substring(0, dotIndex);
		String mapName = filePath + ".map";	
		File file = new File(mapName);
		if (file.exists()) {
			DataInputStream dataStream = null;
			try {
				FileInputStream inputStream = new FileInputStream(file);
				BufferedInputStream bufferedStream = new BufferedInputStream (inputStream);
				dataStream = new DataInputStream (bufferedStream);
				FileLocation fileLocation = new FileLocation();
				String sourceFolder = dataStream.readUTF();
				String relativeFileName = dataStream.readUTF();
				String fileName = sourceFolder + File.separatorChar + relativeFileName;
				fileLocation.fileName = fileName;
				fileLocation.qualifiedName = relativeFileName;
				int count = dataStream.readInt();
				int asLine, javaLine;
				for (int i=0; i<count; i++) {
					asLine = dataStream.readInt();
					javaLine = dataStream.readInt();
					if (lookupLine == asLine) {
						fileLocation.lineNumber = javaLine;
						return fileLocation;
					}
				}
				
			} catch (FileNotFoundException e) {
				e.printStackTrace();
			} catch (IOException e) {
				e.printStackTrace();
			} finally {			
				if (dataStream != null) {
					try {
						dataStream.close();
					} catch (IOException e) {
					}
				}
			}
		}
		return null;
	}
	
	FileLocation findLocation (String localTypeName, String javaName, int matchingLine) {
		String mapName = localTypeName + ".map";
		File file = new File(mapName);
		if (file.exists()) {
			DataInputStream dataStream = null;
			try {
				FileInputStream inputStream = new FileInputStream(file);
				BufferedInputStream bufferedStream = new BufferedInputStream (inputStream);
				dataStream = new DataInputStream (bufferedStream);
				String sourceFolder = dataStream.readUTF();
				String relativeFileName = dataStream.readUTF();
				String fileName = sourceFolder + File.separatorChar + relativeFileName;
				if (fileName.equals(javaName)) {
					int count = dataStream.readInt();
					int asLine, javaLine;
					for (int j=0; j<count; j++) {
						asLine = dataStream.readInt();
						javaLine = dataStream.readInt();
						if (matchingLine == javaLine) {
							FileLocation location = new FileLocation();
							location.fileName = localTypeName + ".as";
							location.qualifiedName = relativeFileName;
							location.lineNumber = asLine;
							return location;
						}
					}
				}
			} catch (FileNotFoundException e) {
				e.printStackTrace();
			} catch (IOException e) {
				e.printStackTrace();
			} finally {			
				if (dataStream != null) {
					try {
						dataStream.close();
					} catch (IOException e) {
					}
				}
			}
		}
		return null;
	}
	
}
