/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.tools.actionscript.build;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.LinkedList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.swt.tools.actionscript.ActionScriptCorePlugin;

public class ActionScriptCompilerManager {
	
	IActionScriptConsole console;
	Pattern errorPattern = Pattern.compile("\\((\\d+)\\)\\: col\\: (\\d+) (.*): ((.*))");
	
	class ActionScriptCompilerReader implements Runnable {
		Thread fcshThread;
		BufferedReader reader;
		ActionScriptCompilerManager manager;
		boolean isErrorStream;
		
		public ActionScriptCompilerReader(ActionScriptCompilerManager manager, InputStream stream, boolean isErrorStream) {
			this.manager = manager;
			reader = new BufferedReader(new InputStreamReader(stream));
			this.isErrorStream = isErrorStream;
		}
		
		public void run() {
			while (this.fcshThread != null) {
				synchronized (this) {
					int c;
					try {
						StringBuffer buffer = new StringBuffer();
						while ((c = reader.read()) != -1) {
							buffer.append((char) c);
							if (c == '\n') {
								manager.appendLine(buffer.toString(), isErrorStream);
								break;
							} else if (buffer.indexOf("(fcsh) ") == 0) {
								manager.appendLine(buffer.toString(), isErrorStream);
								manager.endCommand();
								break;
							}
						}
						
					} catch (IOException e) {
						e.printStackTrace();
					}
				}
			}

		}

		public void startUp() {
			this.fcshThread = new Thread(this, "Flex Compiler Shell Reader Thread");
			this.fcshThread.setDaemon(true);
			this.fcshThread.setPriority(Thread.NORM_PRIORITY - 1);
			this.fcshThread.start();
		}

	}	
	
	class ActionScriptCompilerWriter implements Runnable {

		Thread fcshThread;
		Process fcshProcess;
		PrintWriter writer;
		LinkedList jobs = new LinkedList();
		
		public ActionScriptCompilerWriter(Process fcshProcess) {
			this.fcshProcess = fcshProcess;
			writer = new PrintWriter(new OutputStreamWriter(fcshProcess.getOutputStream()));
		}
		
		public void run() {
			while (this.fcshThread != null) {
				synchronized (this) {
					try {
						if (jobs.isEmpty()) {
							Thread.sleep(50);
						} else {
							String nextCommand = (String) jobs.removeFirst();
							writer.println(nextCommand);
							writer.flush();
						}
					}  catch (InterruptedException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}
				}
			}

		}

		public void startUp() {
			this.fcshThread = new Thread(this, "Flex Compiler Shell Writer Thread");
			this.fcshThread.setDaemon(true);
			this.fcshThread.setPriority(Thread.NORM_PRIORITY - 1);
			this.fcshThread.start();
		}
		
		public void sendCommand(String command) {
			jobs.add(command);
		}
	}

	ActionScriptCompilerReader output_reader;
	ActionScriptCompilerReader error_reader;
	ActionScriptCompilerWriter writer;
	
	String result;
	boolean outOfMemory;
	
	int currentJob;
	
	StringBuffer resultBuffer = new StringBuffer();
	Process fcsh;
	
	public ActionScriptCompilerManager() {
		reset();
	}
	
	public synchronized void reset () {
		//TODO: e4 fix for other platforms
		
		stop();
		
		String fcshLocation = System.getProperty(ActionScriptParticipant.FLEX_SDK) + File.separator + "bin" + File.separator + "fcsh.exe";
		try {
			fcsh = DebugPlugin.exec(new String[]{fcshLocation}, null);
			output_reader = new ActionScriptCompilerReader(this, fcsh.getInputStream(), false);
			writer = new ActionScriptCompilerWriter(fcsh);
			error_reader = new ActionScriptCompilerReader(this, fcsh.getErrorStream(), true);
			output_reader.startUp();
			error_reader.startUp();
			writer.startUp();
		} catch (CoreException e) {
			ActionScriptCorePlugin.getDefault().getLog().log(e.getStatus());
			e.printStackTrace();
		}
	}
	
	public synchronized void stop () {
		if (fcsh != null) fcsh.destroy();
		if (output_reader != null) output_reader = null;
		if (writer != null) writer = null;
		if (error_reader != null) error_reader = null;
	}
	
	public synchronized String sendCommand (String command) {
		result = null;
		outOfMemory = false;
		resultBuffer = new StringBuffer();
		writer.sendCommand (command);
		if (console != null) {
			appendToConsole(command, false);
			appendToConsole(System.getProperty("line.separator"), false);
		}
		while (result == null) {
			try {
				wait();
				if (outOfMemory) return null;
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
		return result;
	}

	public synchronized void appendLine(String line, boolean error) {
		resultBuffer.append(line);
		appendToConsole(line, error);
		if (error) {
		   outOfMemory = checkOutOfMemory(line);
		   if (outOfMemory) {
			   endCommand();
		   } else 
			   parseErrorOutput (line);
		}
	}
	
	private boolean checkOutOfMemory(String line) {
		if (line.startsWith("java.lang.OutOfMemory")) {
			reset ();
			return true;
		}
		return false;
	}

	synchronized void appendToConsole (String line, boolean error) {
		if (console != null) console.appendText(line, error);
	}
	
	public synchronized void endCommand () {
		this.result = resultBuffer.toString();
		notifyAll();
	}
	
	public void setConsole (IActionScriptConsole console){
		this.console = console;
	}
	
	void parseErrorOutput (String line) {
		Matcher matcher = errorPattern.matcher(line);
		if (matcher.find()) {
			String file = line.substring(0, matcher.start());
			IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
			String workspaceRoot = root.getLocation().toOSString();
			if (file.startsWith(workspaceRoot)) {
				String path = file.substring(workspaceRoot.length());
				IResource resource = root.findMember(path);
				if (resource != null) {
					try {
						IMarker newMarker = resource.createMarker(IMarker.PROBLEM);
						newMarker.setAttribute(IMarker.MESSAGE, "[AS]: " + matcher.group(4));
						newMarker.setAttribute(IMarker.SEVERITY, matcher.group(3).equals("Error") ? IMarker.SEVERITY_ERROR : IMarker.SEVERITY_WARNING);
						newMarker.setAttribute(IMarker.LINE_NUMBER, Integer.parseInt(matcher.group(1)));
					} catch (CoreException e) {
						e.printStackTrace();
					}
				}
			}
		}
	}
}