/*******************************************************************************
 * Copyright (c) 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.pde.internal.core.importing;

import org.eclipse.pde.internal.core.importing.provisional.BundleImportDescription;
import org.eclipse.pde.internal.core.importing.provisional.IBundleImporterDelegate;

import java.util.*;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.*;
import org.eclipse.osgi.util.ManifestElement;
import org.eclipse.pde.internal.core.ICoreConstants;
import org.eclipse.pde.internal.core.PDECore;
import org.eclipse.team.core.*;
import org.osgi.framework.BundleException;
import org.osgi.framework.Constants;

/**
 * Handles SCM CVS headers of the following form. Tag and project name can be specified
 * as extra attributes. When a tag is unspecified, the HEAD tag is used. When a project
 * name is unspecified, it is generated by the module name.
 * <pre>
 * scm:cvs&lt;delimiter&gt;&lt;method&gt;&lt;delimiter&gt;path_to_repository&lt;delimiter&gt;module_name[;tag=version][;project=name]
 * scm:psf&lt;delimiter&gt;&lt;method&gt;&lt;delimiter&gt;path_to_repository&lt;delimiter&gt;module_name[;tag=version][;project=name]
 * </pre>
 * @since 3.6
 */
/**
 *
 */
public class CvsBundleImporterDelegate implements IBundleImporterDelegate {

	private static Set SUPPORTED_VALUES;

	private static final String SCM = "scm:"; //$NON-NLS-1$
	private static final String CVS = "cvs"; //$NON-NLS-1$
//	private static final String PSF = "psf"; //$NON-NLS-1$
	private static final String COLON = ":"; //$NON-NLS-1$
	private static final String PIPE = "|"; //$NON-NLS-1$

	private static final String ATTR_TAG = "tag"; //$NON-NLS-1$
	private static final String ATTR_PROJECT = "project"; //$NON-NLS-1$

	private static RepositoryProviderType CVS_PROVIDER_TYPE = RepositoryProviderType.getProviderType("org.eclipse.team.cvs.core.cvsnature"); //$NON-NLS-1$

	static {
		SUPPORTED_VALUES = new HashSet();
		SUPPORTED_VALUES.add(SCM + CVS + COLON);
		SUPPORTED_VALUES.add(SCM + CVS + PIPE);
//		SUPPORTED_VALUES.add(SCM + PSF + COLON);
//		SUPPORTED_VALUES.add(SCM + PSF + PIPE);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.pde.core.project.IBundleImporterDelegate#validateImport(java.util.Map[])
	 */
	public BundleImportDescription[] validateImport(Map[] manifests) {
		BundleImportDescription[] results = new BundleImportDescription[manifests.length];
		if (CVS_PROVIDER_TYPE != null) {
			for (int i = 0; i < manifests.length; i++) {
				Map manifest = manifests[i];
				String value = (String) manifest.get(ICoreConstants.ECLIPSE_SOURCE_REFERENCES);
				if (value != null && value.length() > 8) {
					String prefix = value.substring(0, 8);
					if (SUPPORTED_VALUES.contains(prefix)) {
						try {
							ManifestElement[] elements = ManifestElement.parseHeader(ICoreConstants.ECLIPSE_SOURCE_REFERENCES, value);
							for (int j = 0; j < elements.length; j++) {
								ManifestElement element = elements[j];
								String url = element.getValue();
								String tag = element.getAttribute(ATTR_TAG);
								String project = element.getAttribute(ATTR_PROJECT);
								if (project == null) {
									String bsn = (String) manifests[i].get(Constants.BUNDLE_SYMBOLICNAME);
									if (bsn != null) {
										ManifestElement[] bsnElement = ManifestElement.parseHeader(Constants.BUNDLE_SYMBOLICNAME, bsn);
										project = bsnElement[0].getValue();
									}
								}
								results[i] = createImportDescription(url, manifest, tag, project);
							}
						} catch (BundleException e) {
							PDECore.log(e);
						}
					}
				}
			}
		}
		return results;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.pde.core.importing.IBundleImporterDelegate#performImport(org.eclipse.pde.core.importing.BundleImportDescription[], org.eclipse.core.runtime.IProgressMonitor)
	 */
	public IProject[] performImport(BundleImportDescription[] descriptions, IProgressMonitor monitor) throws CoreException {
		List references = new ArrayList();
		// collect and validate all header values
		for (int i = 0; i < descriptions.length; i++) {
			references.add(createPSF((CvsBundleImportDescription) descriptions[i]));
		}
		// create projects
		if (!references.isEmpty()) {
			SubMonitor subMonitor = SubMonitor.convert(monitor, references.size());
			ProjectSetCapability psfCapability = CVS_PROVIDER_TYPE.getProjectSetCapability();
			if (psfCapability != null) {
				// TODO: specify shell
				psfCapability.addToWorkspace((String[]) references.toArray(new String[references.size()]), new ProjectSetSerializationContext(), subMonitor);
			} else {
				//TODO: error
			}
			subMonitor.done();
		}
		return null;
	}

	/**
	 * Constructs a CVS project set import description from an SCMURL.
	 * 
	 * @param url
	 * @param manifest bundle manifest
	 * @param tag attribute value or <code>null</code>
	 * @param project suggested project name or <code>null</code>
	 * @return corresponding CVS project set reference
	 */
	private BundleImportDescription createImportDescription(String url, Map manifest, String tag, String project) {
		// format is 1.0,protocol:host:cvs folder,project name [, tag]
		String suffix = url.substring(8);
		String sep = url.substring(7, 8);
		String[] pieces = suffix.split(sep);
		String protocol = pieces[0];
		// the last piece is the module name
		String module = pieces[pieces.length - 1];
		// path is the second last piece
		String path = pieces[pieces.length - 2];
		// there might be a user:password:server:port
		int index = pieces.length - 3; // start looking at the third last piece and work backwards
		// if it's a port number, keep moving back
		String server = null;
		try {
			Integer.parseInt(pieces[index]);
			index--;
			server = pieces[index];
		} catch (NumberFormatException e) {
			server = pieces[index];
		}
		// trim any user/password
		index = server.indexOf('@');
		if (index >= 0) {
			server = server.substring(index + 1);
		}
		if (project == null) {
			// use module name as default
			int slash = pieces[3].lastIndexOf('/');
			if (slash >= 0) {
				project = pieces[3].substring(slash + 1);
			} else {
				project = pieces[3];
			}
		}
		return new CvsBundleImportDescription(project, manifest, protocol, server, path, module, tag);
	}

	/**
	 * Constructs a CVS project set import description from an SCMURL.
	 * 
	 * @param url
	 * @param tag attribute value or <code>null</code>
	 * @param project suggested project name or <code>null</code>
	 * @return corresponding CVS project set reference
	 */
	private String createPSF(CvsBundleImportDescription description) {
		// format is 1.0,protocol:host:cvs folder,project name [, tag]
		StringBuffer buffer = new StringBuffer();
		buffer.append("1.0,:"); //$NON-NLS-1$
		buffer.append(description.protocol); // protocol
		buffer.append(COLON);
		buffer.append(description.server);
		buffer.append(COLON);
		buffer.append(description.path);
		buffer.append(',');
		buffer.append(description.module);
		buffer.append(',');
		buffer.append(description.getProject());
		String tag = description.tag;
		if (tag != null) {
			buffer.append(',');
			buffer.append(tag);
		}
		return buffer.toString();
	}

}
