# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
import logging
from datetime import datetime
from typing import Any, Optional, Union
from uuid import UUID

from sqlalchemy.exc import SQLAlchemyError

from superset import db
from superset.commands.base import BaseCommand
from superset.key_value.exceptions import KeyValueCreateFailedError
from superset.key_value.models import KeyValueEntry
from superset.key_value.types import Key, KeyValueCodec, KeyValueResource
from superset.utils.core import get_user_id

logger = logging.getLogger(__name__)


class CreateKeyValueCommand(BaseCommand):
    resource: KeyValueResource
    value: Any
    codec: KeyValueCodec
    key: Optional[Union[int, UUID]]
    expires_on: Optional[datetime]

    def __init__(  # pylint: disable=too-many-arguments
        self,
        resource: KeyValueResource,
        value: Any,
        codec: KeyValueCodec,
        key: Optional[Union[int, UUID]] = None,
        expires_on: Optional[datetime] = None,
    ):
        """
        Create a new key-value pair

        :param resource: the resource (dashboard, chart etc)
        :param value: the value to persist in the key-value store
        :param codec: codec used to encode the value
        :param key: id of entry (autogenerated if undefined)
        :param expires_on: entry expiration time
        :
        """
        self.resource = resource
        self.value = value
        self.codec = codec
        self.key = key
        self.expires_on = expires_on

    def run(self) -> Key:
        """
        Persist the value

        :return: the key associated with the persisted value

        """
        try:
            return self.create()
        except SQLAlchemyError as ex:
            db.session.rollback()
            raise KeyValueCreateFailedError() from ex

    def validate(self) -> None:
        pass

    def create(self) -> Key:
        try:
            value = self.codec.encode(self.value)
        except Exception as ex:
            raise KeyValueCreateFailedError("Unable to encode value") from ex
        entry = KeyValueEntry(
            resource=self.resource.value,
            value=value,
            created_on=datetime.now(),
            created_by_fk=get_user_id(),
            expires_on=self.expires_on,
        )
        if self.key is not None:
            try:
                if isinstance(self.key, UUID):
                    entry.uuid = self.key
                else:
                    entry.id = self.key
            except ValueError as ex:
                raise KeyValueCreateFailedError() from ex
        db.session.add(entry)
        db.session.commit()
        return Key(id=entry.id, uuid=entry.uuid)
