/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.ratis.protocol;

import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.thirdparty.com.google.protobuf.UnsafeByteOperations;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.WeakValueCache;

import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.UUID;
import java.util.function.Supplier;

/** Unique identifier implemented using {@link UUID}. */
public abstract class RaftId {
  static final UUID ZERO_UUID = new UUID(0L, 0L);
  static final ByteString ZERO_UUID_BYTESTRING = toByteString(ZERO_UUID);
  private static final int BYTE_LENGTH = 16;

  static UUID toUuid(ByteString bytes) {
    Objects.requireNonNull(bytes, "bytes == null");
    Preconditions.assertSame(BYTE_LENGTH, bytes.size(), "bytes.size()");
    final ByteBuffer buf = bytes.asReadOnlyByteBuffer();
    return new UUID(buf.getLong(), buf.getLong());
  }

  static ByteString toByteString(UUID uuid) {
    Objects.requireNonNull(uuid, "uuid == null");
    final byte[] array = new byte[BYTE_LENGTH];
    ByteBuffer.wrap(array)
        .putLong(uuid.getMostSignificantBits())
        .putLong(uuid.getLeastSignificantBits());
    return UnsafeByteOperations.unsafeWrap(array);
  }

  abstract static class Factory<ID extends RaftId> {
    private final WeakValueCache<UUID, ID> cache;

    Factory(Class<ID> clazz) {
      this.cache = new WeakValueCache<>(clazz.getSimpleName() + "_UUID", this::newInstance);
    }

    abstract ID newInstance(UUID uuid);

    WeakValueCache<UUID, ID> getCache() {
      return cache;
    }

    final ID valueOf(UUID uuid) {
      return cache.getOrCreate(uuid);
    }

    final ID valueOf(ByteString bytes) {
      return bytes != null? valueOf(toUuid(bytes)): emptyId();
    }

    ID emptyId() {
      return valueOf(ZERO_UUID);
    }

    ID randomId() {
      return valueOf(UUID.randomUUID());
    }
  }

  private final UUID uuid;
  private final Supplier<ByteString> uuidBytes;
  private final Supplier<String> uuidString;

  RaftId(UUID uuid) {
    this.uuid = Objects.requireNonNull(uuid, "uuid == null");
    this.uuidBytes = JavaUtils.memoize(() -> toByteString(uuid));
    this.uuidString = JavaUtils.memoize(() -> createUuidString(uuid));
    Preconditions.assertTrue(ZERO_UUID == uuid || !uuid.equals(ZERO_UUID),
        () -> "Failed to create " + JavaUtils.getClassSimpleName(getClass()) + ": UUID " + ZERO_UUID + " is reserved.");
  }

  /** @return the last 12 hex digits. */
  String createUuidString(UUID id) {
    final String s = id.toString().toUpperCase();
    final int i = s.lastIndexOf('-');
    return s.substring(i + 1);
  }

  public ByteString toByteString() {
    return uuidBytes.get();
  }

  @Override
  public String toString() {
    return uuidString.get();
  }

  @Override
  public boolean equals(Object other) {
    return other == this ||
        (other instanceof RaftId
            && this.getClass() == other.getClass()
            && uuid.equals(((RaftId) other).uuid));
  }

  @Override
  public int hashCode() {
    return uuid.hashCode();
  }

  public UUID getUuid() {
    return uuid;
  }
}
