/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.pinot.broker.routing.instanceselector;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import java.time.Clock;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.annotation.Nullable;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.helix.model.ExternalView;
import org.apache.helix.model.IdealState;
import org.apache.helix.store.zk.ZkHelixPropertyStore;
import org.apache.helix.zookeeper.datamodel.ZNRecord;
import org.apache.pinot.broker.routing.adaptiveserverselector.AdaptiveServerSelector;
import org.apache.pinot.common.assignment.InstancePartitions;
import org.apache.pinot.common.assignment.InstancePartitionsUtils;
import org.apache.pinot.common.metrics.BrokerMetrics;
import org.apache.pinot.core.transport.ServerInstance;
import org.apache.pinot.spi.config.table.TableType;
import org.apache.pinot.spi.config.table.assignment.InstancePartitionsType;
import org.apache.pinot.spi.utils.builder.TableNameBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * Instance selector for multi-stage queries which can ensure that Colocated Tables always leverage Colocated Join
 * whenever possible. To achieve this, this instance-selector uses InstancePartitions (IP) to determine replica-groups,
 * as opposed to IdealState used by other instance-selectors. Moreover, this also uses the requestId generated by
 * Pinot broker to determine the replica-group picked for each table involved in the query, as opposed to using a
 * member variable. There may be scenarios where an instance in the chosen replica-group is down. In that case, this
 * strategy will try to pick another replica-group. For realtime tables, this strategy uses only CONSUMING partitions.
 * This is feature is in <strong>Beta</strong>.
 */
public class MultiStageReplicaGroupSelector extends BaseInstanceSelector {
  private static final Logger LOGGER = LoggerFactory.getLogger(MultiStageReplicaGroupSelector.class);

  private volatile InstancePartitions _instancePartitions;

  public MultiStageReplicaGroupSelector(String tableNameWithType, ZkHelixPropertyStore<ZNRecord> propertyStore,
      BrokerMetrics brokerMetrics, @Nullable AdaptiveServerSelector adaptiveServerSelector, Clock clock,
      boolean useFixedReplica, long newSegmentExpirationTimeInSeconds) {
    super(tableNameWithType, propertyStore, brokerMetrics, adaptiveServerSelector, clock, useFixedReplica,
        newSegmentExpirationTimeInSeconds);
  }

  @Override
  public void init(Set<String> enabledInstances, Map<String, ServerInstance> enabledServerMap, IdealState idealState,
      ExternalView externalView, Set<String> onlineSegments) {
    super.init(enabledInstances, enabledServerMap, idealState, externalView, onlineSegments);
    _instancePartitions = getInstancePartitions();
  }

  @Override
  public void onInstancesChange(Set<String> enabledInstances, List<String> changedInstances) {
    super.onInstancesChange(enabledInstances, changedInstances);
    _instancePartitions = getInstancePartitions();
  }

  @Override
  public void onAssignmentChange(IdealState idealState, ExternalView externalView, Set<String> onlineSegments) {
    super.onAssignmentChange(idealState, externalView, onlineSegments);
    _instancePartitions = getInstancePartitions();
  }

  @Override
  Pair<Map<String, String>, Map<String, String>> select(List<String> segments, int requestId,
      SegmentStates segmentStates, Map<String, String> queryOptions) {
    // Create a copy of InstancePartitions to avoid race-condition with event-listeners above.
    InstancePartitions instancePartitions = _instancePartitions;
    int replicaGroupSelected;
    if (isUseFixedReplica(queryOptions)) {
      // When using sticky routing, we want to iterate over the instancePartitions in order to ensure deterministic
      // selection of replica group across queries i.e. same instance replica group id is picked each time.
      // Since the instances within a selected replica group are iterated in order, the assignment within a selected
      // replica group is guaranteed to be deterministic.
      // Note: This can cause major hotspots in the cluster.
      replicaGroupSelected = 0;
    } else {
      replicaGroupSelected = requestId % instancePartitions.getNumReplicaGroups();
    }

    return assign(Sets.newHashSet(segments), segmentStates, instancePartitions, replicaGroupSelected);
  }

  /**
   * Returns a map from the segmentName to the corresponding server. It tries to select all servers from the
   * preferredReplicaGroup, but if it fails, it will try to select the relevant server from other instance partitions.
   *
   * @return A pair of maps, where the first map contains the segments that are assigned to a server and the second
   * map contains the segments that are optional (i.e., the server is not online to serve that segment).
   * Example:
   * {
   *   "required_segments": {
   *     "segment1": "server1",
   *     "segment2": "server2"
   *   },
   *   "optional_segments": {
   *     "segment3": "server3",
   *     "segment4": "server4"
   *   }
   * }
   */
  private Pair<Map<String, String>, Map<String, String>> assign(Set<String> segments,
    SegmentStates segmentStates, InstancePartitions instancePartitions, int preferredReplicaId) {
    Map<String, Integer> instanceToPartitionMap = instancePartitions.getInstanceToPartitionIdMap();
    Map<String, Set<String>> instanceToSegmentsMap = new HashMap<>();

    // instanceToSegmentsMap stores the mapping from instance to the active segments it can serve.
    for (String segment : segments) {
      List<SegmentInstanceCandidate> candidates = segmentStates.getCandidates(segment);
      Preconditions.checkState(candidates != null && !candidates.isEmpty(),
        "Failed to find servers for segment: %s", segment);
      for (SegmentInstanceCandidate candidate : candidates) {
        instanceToSegmentsMap
          .computeIfAbsent(candidate.getInstance(), k -> new HashSet<>())
          .add(segment);
      }
    }

    // partitionToRequiredSegmentsMap stores the mapping from partition to the segments that need to be served. This
    // is necessary to select appropriate replica group at the instance partition level.
    Map<Integer, Set<String>> partitionToRequiredSegmentsMap = new HashMap<>();
    for (Map.Entry<String, Set<String>> entry : instanceToSegmentsMap.entrySet()) {
      Integer partitionId = instanceToPartitionMap.get(entry.getKey());
      partitionToRequiredSegmentsMap
        .computeIfAbsent(partitionId, k -> new HashSet<>())
        .addAll(entry.getValue());
    }

    // Assign segments to instances based on the partitionToRequiredSegmentsMap. This ensures that we select the
    // appropriate replica group for each set of segments belonging to the same instance partition.
    Map<String, String> segmentToSelectedInstanceMap = new HashMap<>();
    int numPartitions = instancePartitions.getNumPartitions();
    for (int partition = 0; partition < numPartitions; partition++) {
      Set<String> requiredSegments = partitionToRequiredSegmentsMap.get(partition);
      if (requiredSegments != null) {
        getSelectedInstancesForPartition(instanceToSegmentsMap, requiredSegments, partition, preferredReplicaId,
          segmentToSelectedInstanceMap);
      }
    }

    return computeOptionalSegments(segmentToSelectedInstanceMap, segmentStates);
  }

  /**
   * This method selects the instances for the given partition and segments. It iterates over the replica groups
   * in a round-robin fashion, starting from the preferred replica group. It only selects the replica group if all
   * the segments are available in the selected instances.
   * If no replica group is found, it throws an exception.
   */
  private void getSelectedInstancesForPartition(Map<String, Set<String>> instanceToSegmentsMap,
    Set<String> requiredSegments, int partitionId, int preferredReplicaId,
    Map<String, String> segmentToSelectedInstanceMapSink) {
    int numReplicaGroups = _instancePartitions.getNumReplicaGroups();

    for (int replicaGroupOffset = 0; replicaGroupOffset < numReplicaGroups; replicaGroupOffset++) {
      int selectedReplicaGroup = (replicaGroupOffset + preferredReplicaId) % numReplicaGroups;
      List<String> selectedInstances = _instancePartitions.getInstances(partitionId, selectedReplicaGroup);

      Set<String> segmentsFromSelectedInstances = new HashSet<>();
      for (String instance : selectedInstances) {
        Set<String> servedSegments = instanceToSegmentsMap.get(instance);
        if (servedSegments != null) {
          segmentsFromSelectedInstances.addAll(servedSegments);
        }
      }

      if (segmentsFromSelectedInstances.containsAll(requiredSegments)) {
        for (String instance: selectedInstances) {
          Set<String> servedSegments = instanceToSegmentsMap.get(instance);
          if (servedSegments == null || servedSegments.isEmpty()) {
            continue;
          }
          for (String segment : servedSegments) {
            if (requiredSegments.contains(segment)) {
              segmentToSelectedInstanceMapSink.put(segment, instance);
            }
          }
        }
        return; // Successfully selected instances for the partition.
      }
    }

    throw new RuntimeException(
      String.format("Unable to find any replica-group to serve segments in the instance-partition %s for table: %s",
        partitionId, _tableNameWithType));
  }

  /**
   * Based on whether the selected instance for the segment is online to serve that segment or not,
   * this method computes the segments that are optional and the segments that are not.
   */
  private Pair<Map<String, String>, Map<String, String>> computeOptionalSegments(
    Map<String, String> segmentToSelectedInstanceMap, SegmentStates segmentStates) {

    Map<String, String> segmentsToInstanceMap = new HashMap<>();
    Map<String, String> optionalSegmentToInstanceMap = new HashMap<>();

    for (Map.Entry<String, String> entry : segmentToSelectedInstanceMap.entrySet()) {
      String segment = entry.getKey();
      String selectedInstance = entry.getValue();

      List<SegmentInstanceCandidate> candidates = segmentStates.getCandidates(segment);
      // If candidates are null, we will throw an exception and log a warning.
      Preconditions.checkState(candidates != null && !candidates.isEmpty(),
        "Failed to find servers for segment: %s", segment);

      for (SegmentInstanceCandidate candidate : candidates) {
        if (selectedInstance.equals(candidate.getInstance())) {
          // This can only be offline when it is a new segment. And such segment is marked as optional segment so that
          // broker or server can skip it upon any issue to process it.
          if (candidate.isOnline()) {
            segmentsToInstanceMap.put(segment, selectedInstance);
          } else {
            optionalSegmentToInstanceMap.put(segment, selectedInstance);
          }
          break;
        }
      }
    }

    return Pair.of(segmentsToInstanceMap, optionalSegmentToInstanceMap);
  }


  @VisibleForTesting
  protected InstancePartitions getInstancePartitions() {
    // TODO: Evaluate whether we need to provide support for COMPLETE partitions.
    TableType tableType = TableNameBuilder.getTableTypeFromTableName(_tableNameWithType);
    Preconditions.checkNotNull(tableType);
    InstancePartitions instancePartitions;
    if (tableType.equals(TableType.OFFLINE)) {
      instancePartitions = InstancePartitionsUtils.fetchInstancePartitions(_propertyStore,
          InstancePartitionsUtils.getInstancePartitionsName(_tableNameWithType, tableType.name()));
    } else {
      instancePartitions = InstancePartitionsUtils.fetchInstancePartitions(_propertyStore,
          InstancePartitionsUtils.getInstancePartitionsName(_tableNameWithType,
              InstancePartitionsType.CONSUMING.name()));
    }
    Preconditions.checkNotNull(instancePartitions);
    return instancePartitions;
  }
}
