/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * license agreements; and to You under the Apache License, version 2.0:
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * This file is part of the Apache Pekko project, which was derived from Akka.
 */

/*
 * Copyright (C) since 2016 Lightbend Inc. <https://www.lightbend.com>
 */

package org.apache.pekko.stream.connectors.s3.impl.auth

import java.time.{ ZoneId, ZonedDateTime }

import software.amazon.awssdk.auth.credentials._
import software.amazon.awssdk.regions.Region
import org.scalatest.flatspec.AnyFlatSpec
import org.scalatest.matchers.should.Matchers

class SigningKeySpec extends AnyFlatSpec with Matchers {
  behavior.of("A Signing Key")

  val credentials = StaticCredentialsProvider.create(
    AwsBasicCredentials.create("AKIDEXAMPLE", "wJalrXUtnFEMI/K7MDENG+bPxRfiCYEXAMPLEKEY"))

  val signingKey = {
    val requestDate = ZonedDateTime.of(2015, 8, 30, 1, 2, 3, 4, ZoneId.of("UTC"))
    SigningKey(requestDate, credentials, CredentialScope(requestDate.toLocalDate, Region.US_EAST_1, "iam"))
  }

  it should "produce a signing key" in {
    val expected: Array[Byte] = Array(196, 175, 177, 204, 87, 113, 216, 113, 118, 58, 57, 62, 68, 183, 3, 87, 27, 85,
      204, 40, 66, 77, 26, 94, 134, 218, 110, 211, 193, 84, 164, 185).map(_.toByte)

    signingKey.key.getEncoded should equal(expected)
  }

  it should "sign a message" in {
    val sts =
      "AWS4-HMAC-SHA256\n20150830T123600Z\n20150830/us-east-1/iam/aws4_request\nf536975d06c0309214f805bb90ccff089219ecd68b2577efef23edd43b7e1a59"
    signingKey.hexEncodedSignature(sts.getBytes) should equal(
      "5d672d79c15b13162d9279b0855cfba6789a8edb4c82c400e06b5924a6f2b5d7")
  }
}
