/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.zookeeper;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.Abortable;
import org.apache.hadoop.hbase.AuthUtil;
import org.apache.hadoop.hbase.ZooKeeperConnectionException;
import org.apache.hadoop.hbase.zookeeper.PendingWatcher;
import org.apache.hadoop.hbase.zookeeper.RecoverableZooKeeper;
import org.apache.hadoop.hbase.zookeeper.ZKConfig;
import org.apache.hadoop.hbase.zookeeper.ZKListener;
import org.apache.hadoop.hbase.zookeeper.ZKUtil;
import org.apache.hadoop.hbase.zookeeper.ZNodePaths;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Id;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class ZKWatcher
implements Watcher,
Abortable,
Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(ZKWatcher.class);
    private String prefix;
    private String identifier;
    private String quorum;
    private final RecoverableZooKeeper recoverableZooKeeper;
    protected Abortable abortable;
    private boolean aborted = false;
    private final ZNodePaths znodePaths;
    private final List<ZKListener> listeners = new CopyOnWriteArrayList<ZKListener>();
    private CountDownLatch saslLatch = new CountDownLatch(1);
    private final Configuration conf;
    private static final Pattern NAME_PATTERN = Pattern.compile("([^/@]*)(/([^/@]*))?@([^/@]*)");

    public ZKWatcher(Configuration conf, String identifier, Abortable abortable) throws ZooKeeperConnectionException, IOException {
        this(conf, identifier, abortable, false);
    }

    public ZKWatcher(Configuration conf, String identifier, Abortable abortable, boolean canCreateBaseZNode) throws IOException, ZooKeeperConnectionException {
        this(conf, identifier, abortable, canCreateBaseZNode, false);
    }

    public ZKWatcher(Configuration conf, String identifier, Abortable abortable, boolean canCreateBaseZNode, boolean clientZK) throws IOException, ZooKeeperConnectionException {
        this.conf = conf;
        if (clientZK) {
            String clientZkQuorumServers = ZKConfig.getClientZKQuorumServersString((Configuration)conf);
            String serverZkQuorumServers = ZKConfig.getZKQuorumServersString((Configuration)conf);
            if (clientZkQuorumServers != null) {
                if (clientZkQuorumServers.equals(serverZkQuorumServers)) {
                    throw new IllegalArgumentException("The quorum settings for client ZK should be different from those for server");
                }
                this.quorum = clientZkQuorumServers;
            } else {
                this.quorum = serverZkQuorumServers;
            }
        } else {
            this.quorum = ZKConfig.getZKQuorumServersString((Configuration)conf);
        }
        this.prefix = identifier;
        this.identifier = identifier + "0x0";
        this.abortable = abortable;
        this.znodePaths = new ZNodePaths(conf);
        PendingWatcher pendingWatcher = new PendingWatcher();
        this.recoverableZooKeeper = ZKUtil.connect(conf, this.quorum, pendingWatcher, identifier);
        pendingWatcher.prepare(this);
        if (canCreateBaseZNode) {
            try {
                this.createBaseZNodes();
            }
            catch (ZooKeeperConnectionException zce) {
                try {
                    this.recoverableZooKeeper.close();
                }
                catch (InterruptedException ie) {
                    LOG.debug("Encountered InterruptedException when closing {}", (Object)this.recoverableZooKeeper);
                    Thread.currentThread().interrupt();
                }
                throw zce;
            }
        }
    }

    private void createBaseZNodes() throws ZooKeeperConnectionException {
        try {
            ZKUtil.createWithParents(this, this.znodePaths.baseZNode);
            ZKUtil.createAndFailSilent(this, this.znodePaths.rsZNode);
            ZKUtil.createAndFailSilent(this, this.znodePaths.drainingZNode);
            ZKUtil.createAndFailSilent(this, this.znodePaths.tableZNode);
            ZKUtil.createAndFailSilent(this, this.znodePaths.splitLogZNode);
            ZKUtil.createAndFailSilent(this, this.znodePaths.backupMasterAddressesZNode);
            ZKUtil.createAndFailSilent(this, this.znodePaths.masterMaintZNode);
        }
        catch (KeeperException e) {
            throw new ZooKeeperConnectionException(this.prefix("Unexpected KeeperException creating base node"), (Exception)((Object)e));
        }
    }

    public void checkAndSetZNodeAcls() {
        if (!ZKUtil.isSecureZooKeeper(this.getConfiguration())) {
            LOG.info("not a secure deployment, proceeding");
            return;
        }
        try {
            List<ACL> actualAcls = this.recoverableZooKeeper.getAcl(this.znodePaths.baseZNode, new Stat());
            if (!this.isBaseZnodeAclSetup(actualAcls)) {
                LOG.info("setting znode ACLs");
                this.setZnodeAclsRecursive(this.znodePaths.baseZNode);
            }
        }
        catch (KeeperException.NoNodeException nne) {
            return;
        }
        catch (InterruptedException ie) {
            this.interruptedExceptionNoThrow(ie, false);
        }
        catch (IOException | KeeperException e) {
            LOG.warn("Received exception while checking and setting zookeeper ACLs", e);
        }
    }

    private void setZnodeAclsRecursive(String znode) throws KeeperException, InterruptedException {
        List<String> children = this.recoverableZooKeeper.getChildren(znode, false);
        for (String child : children) {
            this.setZnodeAclsRecursive(ZNodePaths.joinZNode((String)znode, (String)child));
        }
        ArrayList<ACL> acls = ZKUtil.createACL(this, znode, true);
        LOG.info("Setting ACLs for znode:{} , acl:{}", (Object)znode, acls);
        this.recoverableZooKeeper.setAcl(znode, acls, -1);
    }

    private boolean isBaseZnodeAclSetup(List<ACL> acls) throws IOException {
        String[] superUsers;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Checking znode ACLs");
        }
        if ((superUsers = this.conf.getStrings("hbase.superuser")) != null && !this.checkACLForSuperUsers(superUsers, acls)) {
            return false;
        }
        String hbaseUser = UserGroupInformation.getCurrentUser().getShortUserName();
        if (acls.isEmpty()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("ACL is empty");
            }
            return false;
        }
        for (ACL acl : acls) {
            int perms = acl.getPerms();
            Id id = acl.getId();
            if (ZooDefs.Ids.ANYONE_ID_UNSAFE.equals((Object)id)) {
                if (perms == 1) continue;
                if (LOG.isDebugEnabled()) {
                    LOG.debug(String.format("permissions for '%s' are not correct: have 0x%x, want 0x%x", id, perms, 1));
                }
                return false;
            }
            if (superUsers != null && ZKWatcher.isSuperUserId(superUsers, id)) {
                if (perms == 31) continue;
                if (LOG.isDebugEnabled()) {
                    LOG.debug(String.format("permissions for '%s' are not correct: have 0x%x, want 0x%x", id, perms, 31));
                }
                return false;
            }
            if ("sasl".equals(id.getScheme())) {
                String name = id.getId();
                Matcher match = NAME_PATTERN.matcher(name);
                if (match.matches()) {
                    name = match.group(1);
                }
                if (name.equals(hbaseUser)) {
                    if (perms == 31) continue;
                    if (LOG.isDebugEnabled()) {
                        LOG.debug(String.format("permissions for '%s' are not correct: have 0x%x, want 0x%x", id, perms, 31));
                    }
                    return false;
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Unexpected shortname in SASL ACL: {}", (Object)id);
                }
                return false;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("unexpected ACL id '{}'", (Object)id);
            }
            return false;
        }
        return true;
    }

    private boolean checkACLForSuperUsers(String[] superUsers, List<ACL> acls) {
        for (String user : superUsers) {
            boolean hasAccess = false;
            if (AuthUtil.isGroupPrincipal((String)user)) continue;
            for (ACL acl : acls) {
                if (!user.equals(acl.getId().getId())) continue;
                if (acl.getPerms() == 31) {
                    hasAccess = true;
                    break;
                }
                if (!LOG.isDebugEnabled()) break;
                LOG.debug(String.format("superuser '%s' does not have correct permissions: have 0x%x, want 0x%x", acl.getId().getId(), acl.getPerms(), 31));
                break;
            }
            if (hasAccess) continue;
            return false;
        }
        return true;
    }

    public static boolean isSuperUserId(String[] superUsers, Id id) {
        for (String user : superUsers) {
            if (AuthUtil.isGroupPrincipal((String)user) || !new Id("sasl", user).equals((Object)id)) continue;
            return true;
        }
        return false;
    }

    public String toString() {
        return this.identifier + ", quorum=" + this.quorum + ", baseZNode=" + this.znodePaths.baseZNode;
    }

    public String prefix(String str) {
        return this.toString() + " " + str;
    }

    public List<String> getMetaReplicaNodes() throws KeeperException {
        List<String> childrenOfBaseNode = ZKUtil.listChildrenNoWatch(this, this.znodePaths.baseZNode);
        ArrayList<String> metaReplicaNodes = new ArrayList<String>(2);
        if (childrenOfBaseNode != null) {
            String pattern = this.conf.get("zookeeper.znode.metaserver", "meta-region-server");
            for (String child : childrenOfBaseNode) {
                if (!child.startsWith(pattern)) continue;
                metaReplicaNodes.add(child);
            }
        }
        return metaReplicaNodes;
    }

    public void registerListener(ZKListener listener) {
        this.listeners.add(listener);
    }

    public void registerListenerFirst(ZKListener listener) {
        this.listeners.add(0, listener);
    }

    public void unregisterListener(ZKListener listener) {
        this.listeners.remove(listener);
    }

    public void unregisterAllListeners() {
        this.listeners.clear();
    }

    public List<ZKListener> getListeners() {
        return new ArrayList<ZKListener>(this.listeners);
    }

    public int getNumberOfListeners() {
        return this.listeners.size();
    }

    public RecoverableZooKeeper getRecoverableZooKeeper() {
        return this.recoverableZooKeeper;
    }

    public void reconnectAfterExpiration() throws IOException, KeeperException, InterruptedException {
        this.recoverableZooKeeper.reconnectAfterExpiration();
    }

    public String getQuorum() {
        return this.quorum;
    }

    public ZNodePaths getZNodePaths() {
        return this.znodePaths;
    }

    public void process(WatchedEvent event) {
        LOG.debug(this.prefix("Received ZooKeeper Event, type=" + event.getType() + ", state=" + event.getState() + ", path=" + event.getPath()));
        switch (event.getType()) {
            case None: {
                this.connectionEvent(event);
                break;
            }
            case NodeCreated: {
                for (ZKListener listener : this.listeners) {
                    listener.nodeCreated(event.getPath());
                }
                break;
            }
            case NodeDeleted: {
                for (ZKListener listener : this.listeners) {
                    listener.nodeDeleted(event.getPath());
                }
                break;
            }
            case NodeDataChanged: {
                for (ZKListener listener : this.listeners) {
                    listener.nodeDataChanged(event.getPath());
                }
                break;
            }
            case NodeChildrenChanged: {
                for (ZKListener listener : this.listeners) {
                    listener.nodeChildrenChanged(event.getPath());
                }
                break;
            }
            default: {
                throw new IllegalStateException("Received event is not valid: " + event.getState());
            }
        }
    }

    private void connectionEvent(WatchedEvent event) {
        switch (event.getState()) {
            case SyncConnected: {
                this.identifier = this.prefix + "-0x" + Long.toHexString(this.recoverableZooKeeper.getSessionId());
                LOG.debug("{} connected", (Object)this.identifier);
                break;
            }
            case Disconnected: {
                LOG.debug(this.prefix("Received Disconnected from ZooKeeper, ignoring"));
                break;
            }
            case Expired: {
                String msg = this.prefix(this.identifier + " received expired from ZooKeeper, aborting");
                if (this.abortable == null) break;
                this.abortable.abort(msg, (Throwable)new KeeperException.SessionExpiredException());
                break;
            }
            case ConnectedReadOnly: 
            case SaslAuthenticated: 
            case AuthFailed: {
                break;
            }
            default: {
                throw new IllegalStateException("Received event is not valid: " + event.getState());
            }
        }
    }

    public void sync(String path) throws KeeperException {
        this.recoverableZooKeeper.sync(path, null, null);
    }

    public void keeperException(KeeperException ke) throws KeeperException {
        LOG.error(this.prefix("Received unexpected KeeperException, re-throwing exception"), (Throwable)ke);
        throw ke;
    }

    public void interruptedException(InterruptedException ie) throws KeeperException {
        this.interruptedExceptionNoThrow(ie, true);
        KeeperException.SystemErrorException keeperException = new KeeperException.SystemErrorException();
        keeperException.initCause((Throwable)ie);
        throw keeperException;
    }

    public void interruptedExceptionNoThrow(InterruptedException ie, boolean throwLater) {
        LOG.debug(this.prefix("Received InterruptedException, will interrupt current thread" + (throwLater ? " and rethrow a SystemErrorException" : "")), (Throwable)ie);
        Thread.currentThread().interrupt();
    }

    @Override
    public void close() {
        try {
            this.recoverableZooKeeper.close();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    public Configuration getConfiguration() {
        return this.conf;
    }

    public void abort(String why, Throwable e) {
        if (this.abortable != null) {
            this.abortable.abort(why, e);
        } else {
            this.aborted = true;
        }
    }

    public boolean isAborted() {
        return this.abortable == null ? this.aborted : this.abortable.isAborted();
    }
}

