/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.nfs.nfs3;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.client.HdfsDataInputStream;
import org.apache.hadoop.hdfs.client.HdfsDataOutputStream;
import org.apache.hadoop.hdfs.nfs.conf.NfsConfiguration;
import org.apache.hadoop.hdfs.nfs.nfs3.AsyncDataService;
import org.apache.hadoop.hdfs.nfs.nfs3.Nfs3Utils;
import org.apache.hadoop.hdfs.nfs.nfs3.OffsetRange;
import org.apache.hadoop.hdfs.nfs.nfs3.RpcProgramNfs3;
import org.apache.hadoop.hdfs.nfs.nfs3.WriteCtx;
import org.apache.hadoop.io.BytesWritable;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.nfs.nfs3.FileHandle;
import org.apache.hadoop.nfs.nfs3.Nfs3Constant;
import org.apache.hadoop.nfs.nfs3.Nfs3FileAttributes;
import org.apache.hadoop.nfs.nfs3.request.WRITE3Request;
import org.apache.hadoop.nfs.nfs3.response.COMMIT3Response;
import org.apache.hadoop.nfs.nfs3.response.WRITE3Response;
import org.apache.hadoop.nfs.nfs3.response.WccAttr;
import org.apache.hadoop.nfs.nfs3.response.WccData;
import org.apache.hadoop.oncrpc.XDR;
import org.apache.hadoop.oncrpc.security.Verifier;
import org.apache.hadoop.oncrpc.security.VerifierNone;
import org.apache.hadoop.security.IdMappingServiceProvider;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.Time;
import org.jboss.netty.channel.Channel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class OpenFileCtx {
    public static final Logger LOG = LoggerFactory.getLogger(OpenFileCtx.class);
    private static long DUMP_WRITE_WATER_MARK = 0x100000L;
    private final DFSClient client;
    private final IdMappingServiceProvider iug;
    private volatile boolean activeState;
    private volatile boolean asyncStatus;
    private volatile long asyncWriteBackStartOffset;
    private AtomicLong nextOffset;
    private final HdfsDataOutputStream fos;
    private final boolean aixCompatMode;
    private Nfs3FileAttributes latestAttr;
    private final ConcurrentNavigableMap<OffsetRange, WriteCtx> pendingWrites;
    private final ConcurrentNavigableMap<Long, CommitCtx> pendingCommits;
    private long lastAccessTime;
    private volatile boolean enabledDump;
    private FileOutputStream dumpOut;
    private AtomicLong nonSequentialWriteInMemory;
    private RandomAccessFile raf;
    private final String dumpFilePath;
    private Daemon dumpThread;
    private final boolean uploadLargeFile;

    private void updateLastAccessTime() {
        this.lastAccessTime = Time.monotonicNow();
    }

    private boolean checkStreamTimeout(long streamTimeout) {
        return Time.monotonicNow() - this.lastAccessTime > streamTimeout;
    }

    long getLastAccessTime() {
        return this.lastAccessTime;
    }

    public long getNextOffset() {
        return this.nextOffset.get();
    }

    boolean getActiveState() {
        return this.activeState;
    }

    boolean hasPendingWork() {
        return this.pendingWrites.size() != 0 || this.pendingCommits.size() != 0;
    }

    private long updateNonSequentialWriteInMemory(long count) {
        long newValue = this.nonSequentialWriteInMemory.addAndGet(count);
        LOG.debug("Update nonSequentialWriteInMemory by {} new value: {}", (Object)count, (Object)newValue);
        Preconditions.checkState((newValue >= 0L ? 1 : 0) != 0, (Object)("nonSequentialWriteInMemory is negative " + newValue + " after update with count " + count));
        return newValue;
    }

    OpenFileCtx(HdfsDataOutputStream fos, Nfs3FileAttributes latestAttr, String dumpFilePath, DFSClient client, IdMappingServiceProvider iug) {
        this(fos, latestAttr, dumpFilePath, client, iug, false, new NfsConfiguration());
    }

    OpenFileCtx(HdfsDataOutputStream fos, Nfs3FileAttributes latestAttr, String dumpFilePath, DFSClient client, IdMappingServiceProvider iug, boolean aixCompatMode, NfsConfiguration config) {
        this.fos = fos;
        this.latestAttr = latestAttr;
        this.aixCompatMode = aixCompatMode;
        this.pendingWrites = new ConcurrentSkipListMap<OffsetRange, WriteCtx>(OffsetRange.ReverseComparatorOnMin);
        this.pendingCommits = new ConcurrentSkipListMap<Long, CommitCtx>();
        this.updateLastAccessTime();
        this.activeState = true;
        this.asyncStatus = false;
        this.asyncWriteBackStartOffset = 0L;
        this.dumpOut = null;
        this.raf = null;
        this.nonSequentialWriteInMemory = new AtomicLong(0L);
        this.dumpFilePath = dumpFilePath;
        this.enabledDump = dumpFilePath != null;
        this.nextOffset = new AtomicLong();
        this.nextOffset.set(latestAttr.getSize());
        assert (this.nextOffset.get() == this.fos.getPos());
        this.dumpThread = null;
        this.client = client;
        this.iug = iug;
        this.uploadLargeFile = config.getBoolean("nfs.large.file.upload", true);
    }

    public Nfs3FileAttributes getLatestAttr() {
        return this.latestAttr;
    }

    private long getFlushedOffset() {
        return this.fos.getPos();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void waitForDump() {
        if (!this.enabledDump) {
            LOG.debug("Do nothing, dump is disabled.");
            return;
        }
        if (this.nonSequentialWriteInMemory.get() < DUMP_WRITE_WATER_MARK) {
            return;
        }
        OpenFileCtx openFileCtx = this;
        synchronized (openFileCtx) {
            if (this.nonSequentialWriteInMemory.get() >= DUMP_WRITE_WATER_MARK) {
                LOG.debug("Asking dumper to dump...");
                if (this.dumpThread == null) {
                    this.dumpThread = new Daemon((Runnable)new Dumper());
                    this.dumpThread.start();
                } else {
                    this.notifyAll();
                }
            }
            while (this.nonSequentialWriteInMemory.get() >= DUMP_WRITE_WATER_MARK) {
                try {
                    this.wait();
                }
                catch (InterruptedException interruptedException) {}
            }
        }
    }

    private WriteCtx checkRepeatedWriteRequest(WRITE3Request request, Channel channel, int xid) {
        OffsetRange range = new OffsetRange(request.getOffset(), request.getOffset() + (long)request.getCount());
        WriteCtx writeCtx = (WriteCtx)this.pendingWrites.get(range);
        if (writeCtx == null) {
            return null;
        }
        if (xid != writeCtx.getXid()) {
            LOG.warn("Got a repeated request, same range, with a different xid: {} xid in old request: {}", (Object)xid, (Object)writeCtx.getXid());
        }
        return writeCtx;
    }

    public void receivedNewWrite(DFSClient dfsClient, WRITE3Request request, Channel channel, int xid, AsyncDataService asyncDataService, IdMappingServiceProvider iug) {
        if (!this.activeState) {
            LOG.info("OpenFileCtx is inactive, fileId: {}", (Object)request.getHandle().dumpFileHandle());
            WccData fileWcc = new WccData(this.latestAttr.getWccAttr(), this.latestAttr);
            WRITE3Response response = new WRITE3Response(5, fileWcc, 0, request.getStableHow(), Nfs3Constant.WRITE_COMMIT_VERF);
            Nfs3Utils.writeChannel(channel, response.serialize(new XDR(), xid, (Verifier)new VerifierNone()), xid);
        } else {
            this.updateLastAccessTime();
            WriteCtx existantWriteCtx = this.checkRepeatedWriteRequest(request, channel, xid);
            if (existantWriteCtx != null) {
                if (!existantWriteCtx.getReplied()) {
                    LOG.debug("Repeated write request which hasn't been served: xid={}, drop it.", (Object)xid);
                } else {
                    LOG.debug("Repeated write request which is already served: xid={}, resend response.", (Object)xid);
                    WccData fileWcc = new WccData(this.latestAttr.getWccAttr(), this.latestAttr);
                    WRITE3Response response = new WRITE3Response(0, fileWcc, request.getCount(), request.getStableHow(), Nfs3Constant.WRITE_COMMIT_VERF);
                    Nfs3Utils.writeChannel(channel, response.serialize(new XDR(), xid, (Verifier)new VerifierNone()), xid);
                }
            } else {
                this.receivedNewWriteInternal(dfsClient, request, channel, xid, asyncDataService, iug);
            }
        }
    }

    @VisibleForTesting
    public static void alterWriteRequest(WRITE3Request request, long cachedOffset) {
        long offset = request.getOffset();
        int count = request.getCount();
        long smallerCount = offset + (long)count - cachedOffset;
        LOG.debug("Got overwrite with appended data [{}-{}), current offset {}, drop the overlapped section [{}-{}) and append new data [{}-{}).", new Object[]{offset, offset + (long)count, cachedOffset, offset, cachedOffset, cachedOffset, offset + (long)count});
        ByteBuffer data = request.getData();
        Preconditions.checkState((data.position() == 0 ? 1 : 0) != 0, (Object)"The write request data has non-zero position");
        data.position((int)(cachedOffset - offset));
        Preconditions.checkState(((long)(data.limit() - data.position()) == smallerCount ? 1 : 0) != 0, (Object)"The write request buffer has wrong limit/position regarding count");
        request.setOffset(cachedOffset);
        request.setCount((int)smallerCount);
    }

    @VisibleForTesting
    private static void trimWriteRequest(WriteCtx writeCtx, long currentOffset) {
        long offset = writeCtx.getOffset();
        if (LOG.isDebugEnabled()) {
            int count = writeCtx.getCount();
            LOG.debug(String.format("Trim request [%d-%d), current offset %d, drop the overlapped section [%d-%d) and write new data [%d-%d)", offset, offset + (long)count, currentOffset, offset, currentOffset, currentOffset, offset + (long)count));
        }
        writeCtx.trimWrite((int)(currentOffset - offset));
    }

    private synchronized WriteCtx addWritesToCache(WRITE3Request request, Channel channel, int xid) {
        WriteCtx oldWriteCtx;
        long offset = request.getOffset();
        int count = request.getCount();
        long cachedOffset = this.nextOffset.get();
        int originalCount = -1;
        LOG.debug("requested offset={} and current offset={}", (Object)offset, (Object)cachedOffset);
        if (offset + (long)count <= cachedOffset) {
            LOG.warn(String.format("Got overwrite [%d-%d) smaller than current offset %d, drop the request.", offset, offset + (long)count, cachedOffset));
            return null;
        }
        if (offset < cachedOffset && offset + (long)count > cachedOffset) {
            LOG.warn(String.format("Got overwrite with appended data [%d-%d), current offset %d, drop the overlapped section [%d-%d) and append new data [%d-%d).", offset, offset + (long)count, cachedOffset, offset, cachedOffset, cachedOffset, offset + (long)count));
            LOG.warn("Modify this write to write only the appended data");
            OpenFileCtx.alterWriteRequest(request, cachedOffset);
            originalCount = count;
            offset = request.getOffset();
            count = request.getCount();
        }
        if (offset < cachedOffset) {
            LOG.warn("(offset,count,nextOffset): ({},{},{})", new Object[]{offset, count, this.nextOffset});
            return null;
        }
        WriteCtx.DataState dataState = offset == cachedOffset ? WriteCtx.DataState.NO_DUMP : WriteCtx.DataState.ALLOW_DUMP;
        WriteCtx writeCtx = new WriteCtx(request.getHandle(), request.getOffset(), request.getCount(), originalCount, request.getStableHow(), request.getData(), channel, xid, false, dataState);
        LOG.debug("Add new write to the list with nextOffset {} and requested offset={}", (Object)cachedOffset, (Object)offset);
        if (writeCtx.getDataState() == WriteCtx.DataState.ALLOW_DUMP) {
            this.updateNonSequentialWriteInMemory(count);
        }
        if ((oldWriteCtx = this.checkRepeatedWriteRequest(request, channel, xid)) == null) {
            this.pendingWrites.put(new OffsetRange(offset, offset + (long)count), writeCtx);
            LOG.debug("New write buffered with xid {} nextOffset {}req offset={} mapsize={}", new Object[]{xid, cachedOffset, offset, this.pendingWrites.size()});
        } else {
            LOG.warn("Got a repeated request, same range, with xid: {} nextOffset {} req offset={}", new Object[]{xid, cachedOffset, offset});
        }
        return writeCtx;
    }

    private void processOverWrite(DFSClient dfsClient, WRITE3Request request, Channel channel, int xid, IdMappingServiceProvider iug) {
        WRITE3Response response;
        WccData wccData = new WccData(this.latestAttr.getWccAttr(), null);
        long offset = request.getOffset();
        int count = request.getCount();
        Nfs3Constant.WriteStableHow stableHow = request.getStableHow();
        long cachedOffset = this.nextOffset.get();
        if (offset + (long)count > cachedOffset) {
            LOG.warn("Treat this jumbo write as a real random write, no support.");
            response = new WRITE3Response(22, wccData, 0, Nfs3Constant.WriteStableHow.UNSTABLE, Nfs3Constant.WRITE_COMMIT_VERF);
        } else {
            LOG.debug("Process perfectOverWrite");
            response = this.processPerfectOverWrite(dfsClient, offset, count, stableHow, request.getData().array(), Nfs3Utils.getFileIdPath(request.getHandle()), wccData, iug);
        }
        this.updateLastAccessTime();
        Nfs3Utils.writeChannel(channel, response.serialize(new XDR(), xid, (Verifier)new VerifierNone()), xid);
    }

    private synchronized boolean checkAndStartWrite(AsyncDataService asyncDataService, WriteCtx writeCtx) {
        if (writeCtx.getOffset() == this.nextOffset.get()) {
            if (!this.asyncStatus) {
                LOG.debug("Trigger the write back task. Current nextOffset: {}", (Object)this.nextOffset.get());
                this.asyncStatus = true;
                this.asyncWriteBackStartOffset = writeCtx.getOffset();
                asyncDataService.execute(new AsyncDataService.WriteBackTask(this));
            } else {
                LOG.debug("The write back thread is working.");
            }
            return true;
        }
        return false;
    }

    private void receivedNewWriteInternal(DFSClient dfsClient, WRITE3Request request, Channel channel, int xid, AsyncDataService asyncDataService, IdMappingServiceProvider iug) {
        Nfs3Constant.WriteStableHow stableHow = request.getStableHow();
        WccAttr preOpAttr = this.latestAttr.getWccAttr();
        int count = request.getCount();
        WriteCtx writeCtx = this.addWritesToCache(request, channel, xid);
        if (writeCtx == null) {
            this.processOverWrite(dfsClient, request, channel, xid, iug);
        } else {
            boolean startWriting = this.checkAndStartWrite(asyncDataService, writeCtx);
            if (!startWriting) {
                this.waitForDump();
                if (stableHow != Nfs3Constant.WriteStableHow.UNSTABLE) {
                    LOG.info("Have to change stable write to unstable write: {}", (Object)request.getStableHow());
                    stableHow = Nfs3Constant.WriteStableHow.UNSTABLE;
                }
                LOG.debug("UNSTABLE write request, send response for offset: {}", (Object)writeCtx.getOffset());
                WccData fileWcc = new WccData(preOpAttr, this.latestAttr);
                WRITE3Response response = new WRITE3Response(0, fileWcc, count, stableHow, Nfs3Constant.WRITE_COMMIT_VERF);
                RpcProgramNfs3.metrics.addWrite(Nfs3Utils.getElapsedTime(writeCtx.startTime));
                Nfs3Utils.writeChannel(channel, response.serialize(new XDR(), xid, (Verifier)new VerifierNone()), xid);
                writeCtx.setReplied(true);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private WRITE3Response processPerfectOverWrite(DFSClient dfsClient, long offset, int count, Nfs3Constant.WriteStableHow stableHow, byte[] data, String path, WccData wccData, IdMappingServiceProvider iug) {
        WRITE3Response response;
        HdfsDataInputStream fis;
        int readCount;
        byte[] readbuffer;
        block11: {
            readbuffer = new byte[count];
            readCount = 0;
            fis = null;
            try {
                this.fos.hsync(EnumSet.of(HdfsDataOutputStream.SyncFlag.UPDATE_LENGTH));
            }
            catch (ClosedChannelException closedException) {
                LOG.info("The FSDataOutputStream has been closed. Continue processing the perfect overwrite.");
            }
            catch (IOException e) {
                LOG.info("hsync failed when processing possible perfect overwrite, path={} error: {}", (Object)path, (Object)e.toString());
                return new WRITE3Response(5, wccData, 0, stableHow, Nfs3Constant.WRITE_COMMIT_VERF);
            }
            try {
                fis = dfsClient.createWrappedInputStream(dfsClient.open(path));
                readCount = fis.read(offset, readbuffer, 0, count);
                if (readCount >= count) break block11;
                LOG.error("Can't read back {} bytes, partial read size: {}", (Object)count, (Object)readCount);
                WRITE3Response e = new WRITE3Response(5, wccData, 0, stableHow, Nfs3Constant.WRITE_COMMIT_VERF);
            }
            catch (IOException e) {
                WRITE3Response wRITE3Response;
                try {
                    LOG.info("Read failed when processing possible perfect overwrite, path={}", (Object)path, (Object)e);
                    wRITE3Response = new WRITE3Response(5, wccData, 0, stableHow, Nfs3Constant.WRITE_COMMIT_VERF);
                }
                catch (Throwable throwable) {
                    IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{fis});
                    throw throwable;
                }
                IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{fis});
                return wRITE3Response;
            }
            IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{fis});
            return e;
        }
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{fis});
        BytesWritable.Comparator comparator = new BytesWritable.Comparator();
        if (comparator.compare(readbuffer, 0, readCount, data, 0, count) != 0) {
            LOG.info("Perfect overwrite has different content");
            response = new WRITE3Response(22, wccData, 0, stableHow, Nfs3Constant.WRITE_COMMIT_VERF);
        } else {
            LOG.info("Perfect overwrite has same content, updating the mtime, then return success");
            Nfs3FileAttributes postOpAttr = null;
            try {
                dfsClient.setTimes(path, Time.monotonicNow(), -1L);
                postOpAttr = Nfs3Utils.getFileAttr(dfsClient, path, iug);
            }
            catch (IOException e) {
                LOG.info("Got error when processing perfect overwrite, path={} error: {}", (Object)path, (Object)e.toString());
                return new WRITE3Response(5, wccData, 0, stableHow, Nfs3Constant.WRITE_COMMIT_VERF);
            }
            wccData.setPostOpAttr(postOpAttr);
            response = new WRITE3Response(0, wccData, count, stableHow, Nfs3Constant.WRITE_COMMIT_VERF);
        }
        return response;
    }

    public COMMIT_STATUS checkCommit(DFSClient dfsClient, long commitOffset, Channel channel, int xid, Nfs3FileAttributes preOpAttr, boolean fromRead) {
        if (!fromRead) {
            Preconditions.checkState((channel != null && preOpAttr != null ? 1 : 0) != 0);
            this.updateLastAccessTime();
        }
        Preconditions.checkState((commitOffset >= 0L ? 1 : 0) != 0);
        COMMIT_STATUS ret = this.checkCommitInternal(commitOffset, channel, xid, preOpAttr, fromRead);
        LOG.debug("Got commit status: {}", (Object)ret.name());
        if (ret == COMMIT_STATUS.COMMIT_DO_SYNC || ret == COMMIT_STATUS.COMMIT_FINISHED) {
            try {
                this.fos.hsync(EnumSet.of(HdfsDataOutputStream.SyncFlag.UPDATE_LENGTH));
                ret = COMMIT_STATUS.COMMIT_FINISHED;
            }
            catch (ClosedChannelException cce) {
                ret = this.pendingWrites.isEmpty() ? COMMIT_STATUS.COMMIT_FINISHED : COMMIT_STATUS.COMMIT_ERROR;
            }
            catch (IOException e) {
                LOG.error("Got stream error during data sync", (Throwable)e);
                ret = COMMIT_STATUS.COMMIT_ERROR;
            }
        }
        return ret;
    }

    @VisibleForTesting
    synchronized boolean checkSequential(long commitOffset, long nextOffset) {
        Preconditions.checkState((commitOffset >= nextOffset ? 1 : 0) != 0, (Object)("commitOffset " + commitOffset + " less than nextOffset " + nextOffset));
        long offset = nextOffset;
        for (OffsetRange range : this.pendingWrites.descendingKeySet()) {
            if (range.getMin() != offset) {
                return false;
            }
            offset = range.getMax();
            if (offset <= commitOffset) continue;
            return true;
        }
        return false;
    }

    private COMMIT_STATUS handleSpecialWait(boolean fromRead, long commitOffset, Channel channel, int xid, Nfs3FileAttributes preOpAttr) {
        if (!fromRead) {
            CommitCtx commitCtx = new CommitCtx(commitOffset, channel, xid, preOpAttr);
            this.pendingCommits.put(commitOffset, commitCtx);
        }
        LOG.debug("return COMMIT_SPECIAL_WAIT");
        return COMMIT_STATUS.COMMIT_SPECIAL_WAIT;
    }

    @VisibleForTesting
    synchronized COMMIT_STATUS checkCommitInternal(long commitOffset, Channel channel, int xid, Nfs3FileAttributes preOpAttr, boolean fromRead) {
        if (!this.activeState) {
            if (this.pendingWrites.isEmpty()) {
                return COMMIT_STATUS.COMMIT_INACTIVE_CTX;
            }
            return COMMIT_STATUS.COMMIT_INACTIVE_WITH_PENDING_WRITE;
        }
        long flushed = this.getFlushedOffset();
        LOG.debug("getFlushedOffset={} commitOffset={} nextOffset={}", new Object[]{flushed, commitOffset, this.nextOffset.get()});
        if (this.pendingWrites.isEmpty()) {
            if (this.aixCompatMode) {
                return COMMIT_STATUS.COMMIT_FINISHED;
            }
            if (flushed < this.nextOffset.get()) {
                LOG.debug("get commit while still writing to the requested offset, with empty queue");
                return this.handleSpecialWait(fromRead, this.nextOffset.get(), channel, xid, preOpAttr);
            }
            return COMMIT_STATUS.COMMIT_FINISHED;
        }
        Preconditions.checkState((flushed <= this.nextOffset.get() ? 1 : 0) != 0, (Object)("flushed " + flushed + " is larger than nextOffset " + this.nextOffset.get()));
        if (this.uploadLargeFile && !this.aixCompatMode) {
            long co;
            long l = co = commitOffset > 0L ? commitOffset : ((OffsetRange)this.pendingWrites.firstEntry().getKey()).getMax() - 1L;
            if (co <= flushed) {
                return COMMIT_STATUS.COMMIT_DO_SYNC;
            }
            if (co < this.nextOffset.get()) {
                LOG.debug("get commit while still writing to the requested offset");
                return this.handleSpecialWait(fromRead, co, channel, xid, preOpAttr);
            }
            if (this.checkSequential(co, this.nextOffset.get())) {
                return this.handleSpecialWait(fromRead, co, channel, xid, preOpAttr);
            }
            LOG.debug("return COMMIT_SPECIAL_SUCCESS");
            return COMMIT_STATUS.COMMIT_SPECIAL_SUCCESS;
        }
        if (commitOffset > 0L) {
            if (this.aixCompatMode) {
                if (commitOffset <= flushed) {
                    return COMMIT_STATUS.COMMIT_DO_SYNC;
                }
            } else {
                if (commitOffset > flushed) {
                    if (!fromRead) {
                        CommitCtx commitCtx = new CommitCtx(commitOffset, channel, xid, preOpAttr);
                        this.pendingCommits.put(commitOffset, commitCtx);
                    }
                    return COMMIT_STATUS.COMMIT_WAIT;
                }
                return COMMIT_STATUS.COMMIT_DO_SYNC;
            }
        }
        Map.Entry key = this.pendingWrites.firstEntry();
        if (!fromRead) {
            long maxOffset = ((OffsetRange)key.getKey()).getMax() - 1L;
            Preconditions.checkState((maxOffset > 0L ? 1 : 0) != 0);
            CommitCtx commitCtx = new CommitCtx(maxOffset, channel, xid, preOpAttr);
            this.pendingCommits.put(maxOffset, commitCtx);
        }
        return COMMIT_STATUS.COMMIT_WAIT;
    }

    public synchronized boolean streamCleanup(FileHandle handle, long streamTimeout) {
        Preconditions.checkState((streamTimeout >= 10000L ? 1 : 0) != 0);
        if (!this.activeState) {
            return true;
        }
        boolean flag = false;
        if (this.checkStreamTimeout(streamTimeout)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("stream can be closed for fileId: {}", (Object)handle.dumpFileHandle());
            }
            flag = true;
        }
        return flag;
    }

    private synchronized WriteCtx offerNextToWrite() {
        if (this.pendingWrites.isEmpty()) {
            LOG.debug("The async write task has no pending writes, fileId: {}", (Object)this.latestAttr.getFileId());
            this.processCommits(this.nextOffset.get());
            this.asyncStatus = false;
            return null;
        }
        Map.Entry lastEntry = this.pendingWrites.lastEntry();
        OffsetRange range = (OffsetRange)lastEntry.getKey();
        WriteCtx toWrite = (WriteCtx)lastEntry.getValue();
        LOG.trace("range.getMin()={} nextOffset={}", (Object)range.getMin(), (Object)this.nextOffset);
        long offset = this.nextOffset.get();
        if (range.getMin() > offset) {
            LOG.debug("The next sequential write has not arrived yet");
            this.processCommits(this.nextOffset.get());
            this.asyncStatus = false;
        } else if (range.getMax() <= offset) {
            LOG.debug("Remove write {} which is already written from the list", (Object)range);
            this.pendingWrites.remove(range);
        } else {
            if (range.getMin() < offset && range.getMax() > offset) {
                LOG.warn("Got an overlapping write {}, nextOffset={}. Remove and trim it", (Object)range, (Object)offset);
                this.pendingWrites.remove(range);
                OpenFileCtx.trimWriteRequest(toWrite, offset);
                this.nextOffset.addAndGet(toWrite.getCount());
                LOG.debug("Change nextOffset (after trim) to {}", (Object)this.nextOffset.get());
                return toWrite;
            }
            LOG.debug("Remove write {} from the list", (Object)range);
            this.pendingWrites.remove(range);
            this.nextOffset.addAndGet(toWrite.getCount());
            LOG.debug("Change nextOffset to {}", (Object)this.nextOffset.get());
            return toWrite;
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void executeWriteBack() {
        OpenFileCtx openFileCtx;
        Preconditions.checkState((boolean)this.asyncStatus, (Object)("openFileCtx has false asyncStatus, fileId: " + this.latestAttr.getFileId()));
        long startOffset = this.asyncWriteBackStartOffset;
        try {
            WriteCtx toWrite;
            while (this.activeState && (toWrite = this.offerNextToWrite()) != null) {
                this.doSingleWrite(toWrite);
                this.updateLastAccessTime();
            }
            if (!this.activeState) {
                LOG.debug("The openFileCtx is not active anymore, fileId: {}", (Object)this.latestAttr.getFileId());
            }
            openFileCtx = this;
        }
        catch (Throwable throwable) {
            OpenFileCtx openFileCtx2 = this;
            synchronized (openFileCtx2) {
                if (startOffset == this.asyncWriteBackStartOffset) {
                    this.asyncStatus = false;
                } else {
                    LOG.info("Another async task is already started before this one is finalized. fileId: {} asyncStatus: {} original startOffset: {} new startOffset: {}. Won't change asyncStatus here.", new Object[]{this.latestAttr.getFileId(), this.asyncStatus, startOffset, this.asyncWriteBackStartOffset});
                }
            }
            throw throwable;
        }
        synchronized (openFileCtx) {
            if (startOffset == this.asyncWriteBackStartOffset) {
                this.asyncStatus = false;
            } else {
                LOG.info("Another async task is already started before this one is finalized. fileId: {} asyncStatus: {} original startOffset: {} new startOffset: {}. Won't change asyncStatus here.", new Object[]{this.latestAttr.getFileId(), this.asyncStatus, startOffset, this.asyncWriteBackStartOffset});
            }
        }
    }

    private void processCommits(long offset) {
        Preconditions.checkState((offset > 0L ? 1 : 0) != 0);
        long flushedOffset = this.getFlushedOffset();
        Map.Entry entry = this.pendingCommits.firstEntry();
        if (entry == null || ((CommitCtx)entry.getValue()).offset > flushedOffset) {
            return;
        }
        int status = 5;
        try {
            this.fos.hsync(EnumSet.of(HdfsDataOutputStream.SyncFlag.UPDATE_LENGTH));
            status = 0;
        }
        catch (ClosedChannelException cce) {
            if (!this.pendingWrites.isEmpty()) {
                LOG.error("Can't sync for fileId: {}. Channel closed with writes pending", (Object)this.latestAttr.getFileId(), (Object)cce);
            }
            status = 5;
        }
        catch (IOException e) {
            LOG.error("Got stream error during data sync: ", (Throwable)e);
            status = 5;
        }
        try {
            this.latestAttr = Nfs3Utils.getFileAttr(this.client, Nfs3Utils.getFileIdPath(this.latestAttr.getFileId()), this.iug);
        }
        catch (IOException e) {
            LOG.error("Can't get new file attr, fileId: " + this.latestAttr.getFileId(), (Throwable)e);
            status = 5;
        }
        if (this.latestAttr.getSize() != offset) {
            LOG.error("After sync, the expect file size: {}, however actual file size is: {}", (Object)offset, (Object)this.latestAttr.getSize());
            status = 5;
        }
        WccData wccData = new WccData(Nfs3Utils.getWccAttr(this.latestAttr), this.latestAttr);
        while (entry != null && ((CommitCtx)entry.getValue()).offset <= flushedOffset) {
            this.pendingCommits.remove(entry.getKey());
            CommitCtx commit = (CommitCtx)entry.getValue();
            COMMIT3Response response = new COMMIT3Response(status, wccData, Nfs3Constant.WRITE_COMMIT_VERF);
            RpcProgramNfs3.metrics.addCommit(Nfs3Utils.getElapsedTime(commit.startTime));
            Nfs3Utils.writeChannelCommit(commit.getChannel(), response.serialize(new XDR(), commit.getXid(), (Verifier)new VerifierNone()), commit.getXid());
            if (LOG.isDebugEnabled()) {
                LOG.debug("FileId: {} Service time: {}ns. Sent response for commit: {}", new Object[]{this.latestAttr.getFileId(), Nfs3Utils.getElapsedTime(commit.startTime), commit});
            }
            entry = this.pendingCommits.firstEntry();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doSingleWrite(WriteCtx writeCtx) {
        Channel channel = writeCtx.getChannel();
        int xid = writeCtx.getXid();
        long offset = writeCtx.getOffset();
        int count = writeCtx.getCount();
        Nfs3Constant.WriteStableHow stableHow = writeCtx.getStableHow();
        FileHandle handle = writeCtx.getHandle();
        if (LOG.isDebugEnabled()) {
            LOG.debug("do write, fileHandle {} offset: {} length: {} stableHow: {}", new Object[]{handle.dumpFileHandle(), offset, count, stableHow.name()});
        }
        try {
            writeCtx.writeData(this.fos);
            RpcProgramNfs3.metrics.incrBytesWritten(writeCtx.getCount());
            long flushedOffset = this.getFlushedOffset();
            if (flushedOffset != offset + (long)count) {
                throw new IOException("output stream is out of sync, pos=" + flushedOffset + " and nextOffset should be" + (offset + (long)count));
            }
            if (writeCtx.getDataState() == WriteCtx.DataState.ALLOW_DUMP) {
                WriteCtx writeCtx2 = writeCtx;
                synchronized (writeCtx2) {
                    if (writeCtx.getDataState() == WriteCtx.DataState.ALLOW_DUMP) {
                        writeCtx.setDataState(WriteCtx.DataState.NO_DUMP);
                        this.updateNonSequentialWriteInMemory(-count);
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("After writing {} at offset {}, updated the memory count, new value: {}", new Object[]{handle.dumpFileHandle(), offset, this.nonSequentialWriteInMemory.get()});
                        }
                    }
                }
            }
            if (!writeCtx.getReplied()) {
                if (stableHow != Nfs3Constant.WriteStableHow.UNSTABLE) {
                    LOG.info("Do sync for stable write: {}", (Object)writeCtx);
                    try {
                        if (stableHow == Nfs3Constant.WriteStableHow.DATA_SYNC) {
                            this.fos.hsync();
                        } else {
                            Preconditions.checkState((stableHow == Nfs3Constant.WriteStableHow.FILE_SYNC ? 1 : 0) != 0, (Object)("Unknown WriteStableHow: " + stableHow));
                            this.fos.hsync(EnumSet.of(HdfsDataOutputStream.SyncFlag.UPDATE_LENGTH));
                        }
                    }
                    catch (IOException e) {
                        LOG.error("hsync failed with writeCtx: {}", (Object)writeCtx, (Object)e);
                        throw e;
                    }
                }
                WccAttr preOpAttr = this.latestAttr.getWccAttr();
                WccData fileWcc = new WccData(preOpAttr, this.latestAttr);
                if (writeCtx.getOriginalCount() != -1) {
                    LOG.warn("Return original count: {} instead of real data count: {}", (Object)writeCtx.getOriginalCount(), (Object)count);
                    count = writeCtx.getOriginalCount();
                }
                WRITE3Response response = new WRITE3Response(0, fileWcc, count, stableHow, Nfs3Constant.WRITE_COMMIT_VERF);
                RpcProgramNfs3.metrics.addWrite(Nfs3Utils.getElapsedTime(writeCtx.startTime));
                Nfs3Utils.writeChannel(channel, response.serialize(new XDR(), xid, (Verifier)new VerifierNone()), xid);
            }
            this.processCommits(writeCtx.getOffset() + (long)writeCtx.getCount());
        }
        catch (IOException e) {
            LOG.error("Error writing to fileHandle {} at offset {} and length {}", new Object[]{handle.dumpFileHandle(), offset, count, e});
            if (!writeCtx.getReplied()) {
                WRITE3Response response = new WRITE3Response(5);
                Nfs3Utils.writeChannel(channel, response.serialize(new XDR(), xid, (Verifier)new VerifierNone()), xid);
            }
            LOG.info("Clean up open file context for fileId: {}", (Object)this.latestAttr.getFileId());
            this.cleanup();
        }
    }

    synchronized void cleanup() {
        if (!this.activeState) {
            LOG.info("Current OpenFileCtx is already inactive, no need to cleanup.");
            return;
        }
        this.activeState = false;
        if (this.dumpThread != null && this.dumpThread.isAlive()) {
            this.dumpThread.interrupt();
            try {
                this.dumpThread.join(3000L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        try {
            if (this.fos != null) {
                this.fos.close();
            }
        }
        catch (IOException e) {
            LOG.info("Can't close stream for fileId: {}, error: {}", (Object)this.latestAttr.getFileId(), (Object)e.toString());
        }
        LOG.info("There are {} pending writes.", (Object)this.pendingWrites.size());
        WccAttr preOpAttr = this.latestAttr.getWccAttr();
        while (!this.pendingWrites.isEmpty()) {
            OffsetRange key = (OffsetRange)this.pendingWrites.firstKey();
            LOG.info("Fail pending write: {}, nextOffset={}", (Object)key, (Object)this.nextOffset.get());
            WriteCtx writeCtx = (WriteCtx)this.pendingWrites.remove(key);
            if (writeCtx.getReplied()) continue;
            WccData fileWcc = new WccData(preOpAttr, this.latestAttr);
            WRITE3Response response = new WRITE3Response(5, fileWcc, 0, writeCtx.getStableHow(), Nfs3Constant.WRITE_COMMIT_VERF);
            Nfs3Utils.writeChannel(writeCtx.getChannel(), response.serialize(new XDR(), writeCtx.getXid(), (Verifier)new VerifierNone()), writeCtx.getXid());
        }
        if (this.dumpOut != null) {
            try {
                this.dumpOut.close();
            }
            catch (IOException e) {
                LOG.error("Failed to close outputstream of dump file {}", (Object)this.dumpFilePath, (Object)e);
            }
            File dumpFile = new File(this.dumpFilePath);
            if (dumpFile.exists() && !dumpFile.delete()) {
                LOG.error("Failed to delete dumpfile: {}", (Object)dumpFile);
            }
        }
        if (this.raf != null) {
            try {
                this.raf.close();
            }
            catch (IOException e) {
                LOG.error("Got exception when closing input stream of dump file.", (Throwable)e);
            }
        }
    }

    @VisibleForTesting
    ConcurrentNavigableMap<OffsetRange, WriteCtx> getPendingWritesForTest() {
        return this.pendingWrites;
    }

    @VisibleForTesting
    ConcurrentNavigableMap<Long, CommitCtx> getPendingCommitsForTest() {
        return this.pendingCommits;
    }

    @VisibleForTesting
    long getNextOffsetForTest() {
        return this.nextOffset.get();
    }

    @VisibleForTesting
    void setNextOffsetForTest(long newValue) {
        this.nextOffset.set(newValue);
    }

    @VisibleForTesting
    void setActiveStatusForTest(boolean activeState) {
        this.activeState = activeState;
    }

    public String toString() {
        return String.format("activeState: %b asyncStatus: %b nextOffset: %d", this.activeState, this.asyncStatus, this.nextOffset.get());
    }

    class Dumper
    implements Runnable {
        Dumper() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void dump() {
            if (OpenFileCtx.this.dumpOut == null) {
                LOG.info("Create dump file: {}", (Object)OpenFileCtx.this.dumpFilePath);
                File dumpFile = new File(OpenFileCtx.this.dumpFilePath);
                try {
                    Dumper dumper = this;
                    synchronized (dumper) {
                        Preconditions.checkState((boolean)dumpFile.createNewFile(), (String)"The dump file should not exist: %s", (Object)OpenFileCtx.this.dumpFilePath);
                        OpenFileCtx.this.dumpOut = new FileOutputStream(dumpFile);
                    }
                }
                catch (IOException e) {
                    LOG.error("Got failure when creating dump stream {}", (Object)OpenFileCtx.this.dumpFilePath, (Object)e);
                    OpenFileCtx.this.enabledDump = false;
                    if (OpenFileCtx.this.dumpOut != null) {
                        try {
                            OpenFileCtx.this.dumpOut.close();
                        }
                        catch (IOException e1) {
                            LOG.error("Can't close dump stream {}", (Object)OpenFileCtx.this.dumpFilePath, (Object)e);
                        }
                    }
                    return;
                }
            }
            if (OpenFileCtx.this.raf == null) {
                try {
                    OpenFileCtx.this.raf = new RandomAccessFile(OpenFileCtx.this.dumpFilePath, "r");
                }
                catch (FileNotFoundException e) {
                    LOG.error("Can't get random access to file {}", (Object)OpenFileCtx.this.dumpFilePath);
                    OpenFileCtx.this.enabledDump = false;
                    return;
                }
            }
            LOG.debug("Start dump. Before dump, nonSequentialWriteInMemory == {}", (Object)OpenFileCtx.this.nonSequentialWriteInMemory.get());
            Iterator it = OpenFileCtx.this.pendingWrites.keySet().iterator();
            while (OpenFileCtx.this.activeState && it.hasNext() && OpenFileCtx.this.nonSequentialWriteInMemory.get() > 0L) {
                OffsetRange key = (OffsetRange)it.next();
                WriteCtx writeCtx = (WriteCtx)OpenFileCtx.this.pendingWrites.get(key);
                if (writeCtx == null) continue;
                try {
                    long dumpedDataSize = writeCtx.dumpData(OpenFileCtx.this.dumpOut, OpenFileCtx.this.raf);
                    if (dumpedDataSize <= 0L) continue;
                    OpenFileCtx.this.updateNonSequentialWriteInMemory(-dumpedDataSize);
                }
                catch (IOException e) {
                    LOG.error("Dump data failed: {} OpenFileCtx state: {}", new Object[]{writeCtx, OpenFileCtx.this.activeState, e});
                    OpenFileCtx.this.enabledDump = false;
                    return;
                }
            }
            LOG.debug("After dump, nonSequentialWriteInMemory == {}", (Object)OpenFileCtx.this.nonSequentialWriteInMemory.get());
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            while (OpenFileCtx.this.activeState && OpenFileCtx.this.enabledDump) {
                try {
                    if (OpenFileCtx.this.nonSequentialWriteInMemory.get() >= DUMP_WRITE_WATER_MARK) {
                        this.dump();
                    }
                    OpenFileCtx openFileCtx = OpenFileCtx.this;
                    synchronized (openFileCtx) {
                        if (OpenFileCtx.this.nonSequentialWriteInMemory.get() < DUMP_WRITE_WATER_MARK) {
                            OpenFileCtx.this.notifyAll();
                            try {
                                OpenFileCtx.this.wait();
                                LOG.debug("Dumper woke up");
                            }
                            catch (InterruptedException e) {
                                LOG.info("Dumper is interrupted, dumpFilePath = {}", (Object)OpenFileCtx.this.dumpFilePath);
                            }
                        }
                    }
                    LOG.debug("Dumper checking OpenFileCtx activeState: {} enabledDump: {}", (Object)OpenFileCtx.this.activeState, (Object)OpenFileCtx.this.enabledDump);
                }
                catch (Throwable t) {
                    OpenFileCtx openFileCtx = OpenFileCtx.this;
                    synchronized (openFileCtx) {
                        OpenFileCtx.this.notifyAll();
                    }
                    LOG.info("Dumper got Throwable. dumpFilePath: {}", (Object)OpenFileCtx.this.dumpFilePath, (Object)t);
                    OpenFileCtx.this.activeState = false;
                }
            }
        }
    }

    static class CommitCtx {
        private final long offset;
        private final Channel channel;
        private final int xid;
        private final Nfs3FileAttributes preOpAttr;
        public final long startTime;

        long getOffset() {
            return this.offset;
        }

        Channel getChannel() {
            return this.channel;
        }

        int getXid() {
            return this.xid;
        }

        Nfs3FileAttributes getPreOpAttr() {
            return this.preOpAttr;
        }

        long getStartTime() {
            return this.startTime;
        }

        CommitCtx(long offset, Channel channel, int xid, Nfs3FileAttributes preOpAttr) {
            this.offset = offset;
            this.channel = channel;
            this.xid = xid;
            this.preOpAttr = preOpAttr;
            this.startTime = System.nanoTime();
        }

        public String toString() {
            return String.format("offset: %d xid: %d startTime: %d", this.offset, this.xid, this.startTime);
        }
    }

    static enum COMMIT_STATUS {
        COMMIT_FINISHED,
        COMMIT_WAIT,
        COMMIT_INACTIVE_CTX,
        COMMIT_INACTIVE_WITH_PENDING_WRITE,
        COMMIT_ERROR,
        COMMIT_DO_SYNC,
        COMMIT_SPECIAL_WAIT,
        COMMIT_SPECIAL_SUCCESS;

    }
}

