/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.writer;

import com.codahale.metrics.Meter;
import com.codahale.metrics.Timer;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.Lists;
import com.google.common.io.Closer;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import java.io.Closeable;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import org.apache.gobblin.ack.Ackable;
import org.apache.gobblin.configuration.State;
import org.apache.gobblin.exception.NonTransientException;
import org.apache.gobblin.instrumented.Instrumentable;
import org.apache.gobblin.instrumented.Instrumented;
import org.apache.gobblin.metrics.GobblinMetrics;
import org.apache.gobblin.metrics.MetricContext;
import org.apache.gobblin.metrics.Tag;
import org.apache.gobblin.source.extractor.CheckpointableWatermark;
import org.apache.gobblin.stream.RecordEnvelope;
import org.apache.gobblin.util.ConfigUtils;
import org.apache.gobblin.util.ExecutorsUtils;
import org.apache.gobblin.util.FinalState;
import org.apache.gobblin.writer.AsyncDataWriter;
import org.apache.gobblin.writer.DataWriter;
import org.apache.gobblin.writer.WatermarkAwareWriter;
import org.apache.gobblin.writer.WriteCallback;
import org.apache.gobblin.writer.WriteResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AsyncWriterManager<D>
implements WatermarkAwareWriter<D>,
DataWriter<D>,
Instrumentable,
Closeable,
FinalState {
    private static final long MILLIS_TO_NANOS = 1000000L;
    public static final long COMMIT_TIMEOUT_MILLIS_DEFAULT = 60000L;
    public static final long COMMIT_STEP_WAITTIME_MILLIS_DEFAULT = 500L;
    public static final double FAILURE_ALLOWANCE_RATIO_DEFAULT = 0.0;
    public static final boolean RETRIES_ENABLED_DEFAULT = true;
    public static final int NUM_RETRIES_DEFAULT = 5;
    public static final int MIN_RETRY_INTERVAL_MILLIS_DEFAULT = 3;
    public static final int MAX_OUTSTANDING_WRITES_DEFAULT = 1000;
    private final boolean instrumentationEnabled;
    private MetricContext metricContext;
    protected final Closer closer;
    @VisibleForTesting
    Meter recordsAttempted;
    @VisibleForTesting
    Meter recordsIn;
    @VisibleForTesting
    Meter recordsSuccess;
    @VisibleForTesting
    Meter recordsFailed;
    @VisibleForTesting
    Meter bytesWritten;
    @VisibleForTesting
    Optional<Timer> dataWriterTimer;
    private final long commitTimeoutMillis;
    private final long commitStepWaitTimeMillis;
    private final double failureAllowanceRatio;
    private final AsyncDataWriter asyncDataWriter;
    private final int numRetries;
    private final int minRetryIntervalMillis;
    private final Optional<ScheduledThreadPoolExecutor> retryThreadPool;
    private final Logger log;
    @VisibleForTesting
    final Optional<LinkedBlockingQueue<Attempt>> retryQueue;
    private final int maxOutstandingWrites;
    private final Semaphore writePermits;
    private volatile Throwable cachedWriteException = null;

    @Override
    public boolean isWatermarkCapable() {
        return true;
    }

    @Override
    public Map<String, CheckpointableWatermark> getCommittableWatermark() {
        throw new UnsupportedOperationException("This writer does not keep track of committed watermarks");
    }

    @Override
    public Map<String, CheckpointableWatermark> getUnacknowledgedWatermark() {
        throw new UnsupportedOperationException("This writer does not keep track of uncommitted watermarks");
    }

    @Override
    public void switchMetricContext(List<Tag<?>> tags) {
        this.metricContext = (MetricContext)this.closer.register((Closeable)Instrumented.newContextFromReferenceContext(this.metricContext, tags, (Optional<String>)Optional.absent()));
        this.regenerateMetrics();
    }

    @Override
    public void switchMetricContext(MetricContext context) {
        this.metricContext = context;
        this.regenerateMetrics();
    }

    @Override
    public List<Tag<?>> generateTags(State state) {
        return Lists.newArrayList();
    }

    @Override
    @Nonnull
    public MetricContext getMetricContext() {
        return this.metricContext;
    }

    @Override
    public boolean isInstrumentationEnabled() {
        return this.instrumentationEnabled;
    }

    public State getFinalState() {
        return new State();
    }

    protected void regenerateMetrics() {
        this.recordsIn = this.metricContext.meter("gobblin.writer.records.in");
        this.recordsAttempted = this.metricContext.meter("gobblin.writer.records.attempted");
        this.recordsSuccess = this.metricContext.meter("gobblin.writer.successful.writes");
        this.recordsFailed = this.metricContext.meter("gobblin.writer.failed.writes");
        this.bytesWritten = this.metricContext.meter("gobblin.writer.bytes.written");
        this.dataWriterTimer = this.isInstrumentationEnabled() ? Optional.of((Object)this.metricContext.timer("gobblin.writer.write.time")) : Optional.absent();
    }

    protected AsyncWriterManager(Config config, long commitTimeoutMillis, long commitStepWaitTimeMillis, double failureAllowanceRatio, boolean retriesEnabled, int numRetries, int minRetryIntervalMillis, int maxOutstandingWrites, AsyncDataWriter asyncDataWriter, Optional<Logger> loggerOptional) {
        Preconditions.checkArgument((commitTimeoutMillis > 0L ? 1 : 0) != 0, (Object)"Commit timeout must be greater than 0");
        Preconditions.checkArgument((commitStepWaitTimeMillis > 0L ? 1 : 0) != 0, (Object)"Commit step wait time must be greater than 0");
        Preconditions.checkArgument((commitStepWaitTimeMillis < commitTimeoutMillis ? 1 : 0) != 0, (Object)"Commit step wait time must be less than commit timeout");
        Preconditions.checkArgument((failureAllowanceRatio <= 1.0 && failureAllowanceRatio >= 0.0 ? 1 : 0) != 0, (Object)"Failure Allowance must be a ratio between 0 and 1");
        Preconditions.checkArgument((maxOutstandingWrites > 0 ? 1 : 0) != 0, (Object)"Max outstanding writes must be greater than 0");
        Preconditions.checkNotNull((Object)asyncDataWriter, (Object)"Async Data Writer cannot be null");
        this.log = loggerOptional.isPresent() ? (Logger)loggerOptional.get() : LoggerFactory.getLogger(AsyncWriterManager.class);
        this.closer = Closer.create();
        State state = ConfigUtils.configToState((Config)config);
        this.instrumentationEnabled = GobblinMetrics.isEnabled((State)state);
        this.metricContext = (MetricContext)this.closer.register((Closeable)Instrumented.getMetricContext(state, asyncDataWriter.getClass()));
        this.regenerateMetrics();
        this.commitTimeoutMillis = commitTimeoutMillis;
        this.commitStepWaitTimeMillis = commitStepWaitTimeMillis;
        this.failureAllowanceRatio = failureAllowanceRatio;
        this.minRetryIntervalMillis = minRetryIntervalMillis;
        if (retriesEnabled) {
            this.numRetries = numRetries;
            this.retryQueue = Optional.of(new LinkedBlockingQueue());
            this.retryThreadPool = Optional.of((Object)new ScheduledThreadPoolExecutor(1, ExecutorsUtils.newDaemonThreadFactory((Optional)Optional.of((Object)this.log), (Optional)Optional.of((Object)"AsyncWriteManagerRetry-%d"))));
            ((ScheduledThreadPoolExecutor)this.retryThreadPool.get()).execute(new RetryRunner());
        } else {
            this.numRetries = 0;
            this.retryQueue = Optional.absent();
            this.retryThreadPool = Optional.absent();
        }
        this.maxOutstandingWrites = maxOutstandingWrites;
        this.writePermits = new Semaphore(maxOutstandingWrites);
        this.asyncDataWriter = asyncDataWriter;
        this.closer.register((Closeable)asyncDataWriter);
    }

    @Override
    public void writeEnvelope(RecordEnvelope<D> recordEnvelope) throws IOException {
        this.write((D)recordEnvelope.getRecord(), (Ackable)recordEnvelope);
    }

    public void write(D record) throws IOException {
        this.write(record, Ackable.NoopAckable);
    }

    private void write(D record, Ackable ackable) throws IOException {
        this.maybeThrow();
        int spinNum = 0;
        try {
            while (!this.writePermits.tryAcquire(100L, TimeUnit.MILLISECONDS)) {
                if (++spinNum % 50 != 0) continue;
                this.log.info("Spinning due to pending writes, in = " + this.recordsIn.getCount() + ", success = " + this.recordsSuccess.getCount() + ", failed = " + this.recordsFailed.getCount() + ", maxOutstandingWrites = " + this.maxOutstandingWrites);
            }
        }
        catch (InterruptedException e) {
            Throwables.propagate((Throwable)e);
        }
        this.recordsIn.mark();
        this.attemptWrite(new Attempt(record, ackable));
    }

    private boolean isFailureFatal() {
        return this.failureAllowanceRatio == 0.0;
    }

    private void makeNextWriteThrow(Throwable t) {
        this.log.error("Will make next write throw", t);
        this.cachedWriteException = t;
    }

    private void maybeThrow() {
        if (this.cachedWriteException != null) {
            throw new NonTransientException("Irrecoverable failure on async write", this.cachedWriteException);
        }
    }

    private void attemptWrite(final Attempt attempt) {
        this.recordsAttempted.mark();
        attempt.setPrevAttemptTimestampNanos(System.nanoTime());
        this.asyncDataWriter.write(attempt.record, new WriteCallback<Object>(){

            @Override
            public void onSuccess(WriteResponse writeResponse) {
                try {
                    attempt.ackable.ack();
                    AsyncWriterManager.this.recordsSuccess.mark();
                    if (writeResponse.bytesWritten() > 0L) {
                        AsyncWriterManager.this.bytesWritten.mark(writeResponse.bytesWritten());
                    }
                    if (AsyncWriterManager.this.dataWriterTimer.isPresent()) {
                        ((Timer)AsyncWriterManager.this.dataWriterTimer.get()).update(System.nanoTime() - attempt.getPrevAttemptTimestampNanos(), TimeUnit.NANOSECONDS);
                    }
                }
                finally {
                    AsyncWriterManager.this.writePermits.release();
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void onFailure(Throwable throwable) {
                long currTime = System.nanoTime();
                if (AsyncWriterManager.this.dataWriterTimer.isPresent()) {
                    ((Timer)AsyncWriterManager.this.dataWriterTimer.get()).update(currTime - attempt.getPrevAttemptTimestampNanos(), TimeUnit.NANOSECONDS);
                }
                if (attempt.attemptNum <= AsyncWriterManager.this.numRetries) {
                    attempt.incAttempt();
                    attempt.setPrevAttemptFailure(throwable);
                    ((LinkedBlockingQueue)AsyncWriterManager.this.retryQueue.get()).add(attempt);
                } else {
                    try {
                        AsyncWriterManager.this.recordsFailed.mark();
                        AsyncWriterManager.this.log.debug("Failed to write record : {}", (Object)attempt.getRecord().toString(), (Object)throwable);
                        if (AsyncWriterManager.this.isFailureFatal()) {
                            AsyncWriterManager.this.makeNextWriteThrow(throwable);
                        } else {
                            attempt.ackable.ack();
                        }
                    }
                    finally {
                        AsyncWriterManager.this.writePermits.release();
                    }
                }
            }
        });
    }

    public void cleanup() throws IOException {
    }

    public long recordsWritten() {
        return this.recordsSuccess.getCount();
    }

    public long bytesWritten() throws IOException {
        return this.bytesWritten.getCount();
    }

    @Override
    public void close() throws IOException {
        this.log.info("Close called");
        this.closer.close();
        if (this.retryThreadPool.isPresent()) {
            ExecutorsUtils.shutdownExecutorService((ExecutorService)((ExecutorService)this.retryThreadPool.get()), (Optional)Optional.of((Object)this.log), (long)1L, (TimeUnit)TimeUnit.MILLISECONDS);
        }
        this.log.info("Successfully done closing");
    }

    public void commit() throws IOException {
        this.log.info("Commit called, will wait for commitTimeout : {} ms", (Object)this.commitTimeoutMillis);
        long commitTimeoutNanos = this.commitTimeoutMillis * 1000000L;
        long commitStartTime = System.nanoTime();
        this.asyncDataWriter.flush();
        while (System.nanoTime() - commitStartTime < commitTimeoutNanos && this.recordsIn.getCount() != this.recordsSuccess.getCount() + this.recordsFailed.getCount()) {
            this.log.debug("Commit waiting... records produced: {}, written: {}, failed: {}", new Object[]{this.recordsIn.getCount(), this.recordsSuccess.getCount(), this.recordsFailed.getCount()});
            try {
                Thread.sleep(this.commitStepWaitTimeMillis);
            }
            catch (InterruptedException e) {
                this.log.info("Interrupted while waiting for commit to complete");
                throw new IOException("Interrupted while waiting for commit to complete", e);
            }
        }
        this.log.debug("Commit done waiting");
        long recordsProducedFinal = this.recordsIn.getCount();
        long recordsWrittenFinal = this.recordsSuccess.getCount();
        long recordsFailedFinal = this.recordsFailed.getCount();
        long unacknowledgedWrites = recordsProducedFinal - recordsWrittenFinal - recordsFailedFinal;
        long totalFailures = unacknowledgedWrites + recordsFailedFinal;
        if (unacknowledgedWrites > 0L) {
            this.log.warn("Timeout waiting for all writes to be acknowledged. Missing {} responses out of {}", (Object)unacknowledgedWrites, (Object)recordsProducedFinal);
        }
        if (totalFailures > 0L && recordsProducedFinal > 0L) {
            this.log.info("Commit failed to write {} records ({} failed, {} unacknowledged) out of {} produced", new Object[]{totalFailures, recordsFailedFinal, unacknowledgedWrites, recordsProducedFinal});
            double failureRatio = (double)totalFailures / (double)recordsProducedFinal;
            if (failureRatio > this.failureAllowanceRatio) {
                this.log.error("Aborting because this is greater than the failureAllowance percentage: {}", (Object)(this.failureAllowanceRatio * 100.0));
                throw new IOException("Failed to meet failureAllowance SLA", this.cachedWriteException);
            }
            this.log.warn("Committing because the observed failure percentage {} is less than the failureAllowance percentage: {}", (Object)(failureRatio * 100.0), (Object)(this.failureAllowanceRatio * 100.0));
        }
        this.log.info("Successfully committed {} records.", (Object)recordsWrittenFinal);
    }

    public void flush() throws IOException {
        this.asyncDataWriter.flush();
    }

    public static AsyncWriterManagerBuilder builder() {
        return new AsyncWriterManagerBuilder();
    }

    public static class AsyncWriterManagerBuilder {
        private Config config = ConfigFactory.empty();
        private long commitTimeoutMillis = 60000L;
        private long commitStepWaitTimeMillis = 500L;
        private double failureAllowanceRatio = 0.0;
        private boolean retriesEnabled = true;
        private int numRetries = 5;
        private int maxOutstandingWrites = 1000;
        private AsyncDataWriter asyncDataWriter;
        private Optional<Logger> logger = Optional.absent();

        public AsyncWriterManagerBuilder config(Config config) {
            this.config = config;
            return this;
        }

        public AsyncWriterManagerBuilder commitTimeoutMillis(long commitTimeoutMillis) {
            this.commitTimeoutMillis = commitTimeoutMillis;
            return this;
        }

        public AsyncWriterManagerBuilder commitStepWaitTimeInMillis(long commitStepWaitTimeMillis) {
            this.commitStepWaitTimeMillis = commitStepWaitTimeMillis;
            return this;
        }

        public AsyncWriterManagerBuilder failureAllowanceRatio(double failureAllowanceRatio) {
            Preconditions.checkArgument((failureAllowanceRatio <= 1.0 && failureAllowanceRatio >= 0.0 ? 1 : 0) != 0, (Object)"Failure Allowance must be a ratio between 0 and 1");
            this.failureAllowanceRatio = failureAllowanceRatio;
            return this;
        }

        public AsyncWriterManagerBuilder asyncDataWriter(AsyncDataWriter asyncDataWriter) {
            this.asyncDataWriter = asyncDataWriter;
            return this;
        }

        public AsyncWriterManagerBuilder retriesEnabled(boolean retriesEnabled) {
            this.retriesEnabled = retriesEnabled;
            return this;
        }

        public AsyncWriterManagerBuilder numRetries(int numRetries) {
            this.numRetries = numRetries;
            return this;
        }

        public AsyncWriterManagerBuilder maxOutstandingWrites(int maxOutstandingWrites) {
            this.maxOutstandingWrites = maxOutstandingWrites;
            return this;
        }

        public AsyncWriterManagerBuilder logger(Optional<Logger> logger) {
            this.logger = logger;
            return this;
        }

        public AsyncWriterManager build() {
            return new AsyncWriterManager(this.config, this.commitTimeoutMillis, this.commitStepWaitTimeMillis, this.failureAllowanceRatio, this.retriesEnabled, this.numRetries, 3, this.maxOutstandingWrites, this.asyncDataWriter, this.logger);
        }
    }

    private class RetryRunner
    implements Runnable {
        private final LinkedBlockingQueue<Attempt> retryQueue;
        private final long minRetryIntervalNanos;

        public RetryRunner() {
            Preconditions.checkArgument((boolean)AsyncWriterManager.this.retryQueue.isPresent(), (Object)"RetryQueue must be present for RetryRunner");
            this.retryQueue = (LinkedBlockingQueue)AsyncWriterManager.this.retryQueue.get();
            this.minRetryIntervalNanos = (long)AsyncWriterManager.this.minRetryIntervalMillis * 1000000L;
        }

        private void maybeSleep(long lastAttemptTimestampNanos) throws InterruptedException {
            long timeDiff = System.nanoTime() - lastAttemptTimestampNanos;
            long timeToSleep = this.minRetryIntervalNanos - timeDiff;
            if (timeToSleep > 0L) {
                Thread.sleep(timeToSleep / 1000000L);
            }
        }

        @Override
        public void run() {
            while (true) {
                try {
                    while (true) {
                        Attempt attempt;
                        if ((attempt = this.retryQueue.take()) == null) {
                            continue;
                        }
                        this.maybeSleep(attempt.getPrevAttemptTimestampNanos());
                        AsyncWriterManager.this.attemptWrite(attempt);
                    }
                }
                catch (InterruptedException e) {
                    AsyncWriterManager.this.log.info("Retry thread interrupted... will exit");
                    Throwables.propagate((Throwable)e);
                    continue;
                }
                break;
            }
        }
    }

    class Attempt {
        private final D record;
        private final Ackable ackable;
        private int attemptNum;
        private Throwable prevAttemptFailure;
        private long prevAttemptTimestampNanos;

        void incAttempt() {
            ++this.attemptNum;
        }

        Attempt(D record, Ackable ackable) {
            this.record = record;
            this.ackable = ackable;
            this.attemptNum = 1;
            this.prevAttemptFailure = null;
            this.prevAttemptTimestampNanos = -1L;
        }

        public D getRecord() {
            return this.record;
        }

        public Ackable getAckable() {
            return this.ackable;
        }

        public int getAttemptNum() {
            return this.attemptNum;
        }

        public Throwable getPrevAttemptFailure() {
            return this.prevAttemptFailure;
        }

        public long getPrevAttemptTimestampNanos() {
            return this.prevAttemptTimestampNanos;
        }

        public void setPrevAttemptFailure(Throwable prevAttemptFailure) {
            this.prevAttemptFailure = prevAttemptFailure;
        }

        public void setPrevAttemptTimestampNanos(long prevAttemptTimestampNanos) {
            this.prevAttemptTimestampNanos = prevAttemptTimestampNanos;
        }
    }
}

