// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.trees.expressions.functions.agg;

import org.apache.doris.catalog.FunctionSignature;
import org.apache.doris.nereids.exceptions.AnalysisException;
import org.apache.doris.nereids.trees.expressions.Expression;
import org.apache.doris.nereids.trees.expressions.functions.AlwaysNotNullable;
import org.apache.doris.nereids.trees.expressions.functions.ExplicitlyCastableSignature;
import org.apache.doris.nereids.trees.expressions.visitor.ExpressionVisitor;
import org.apache.doris.nereids.types.ArrayType;
import org.apache.doris.nereids.types.BooleanType;
import org.apache.doris.nereids.util.ExpressionUtils;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;

import java.util.List;

/**
 * AggregateFunction 'retention'. This class is generated by GenerateFunction.
 */
public class Retention extends AggregateFunction
        implements ExplicitlyCastableSignature, AlwaysNotNullable {

    public static final List<FunctionSignature> SIGNATURES = ImmutableList.of(
            FunctionSignature.ret(ArrayType.of(BooleanType.INSTANCE)).varArgs(BooleanType.INSTANCE)
    );

    /**
     * constructor with 1 or more arguments.
     */
    public Retention(Expression arg, Expression... varArgs) {
        super("retention", ExpressionUtils.mergeArguments(arg, varArgs));
    }

    /**
     * constructor with 1 or more arguments.
     */
    public Retention(boolean distinct, Expression arg, Expression... varArgs) {
        super("retention", distinct,
                ExpressionUtils.mergeArguments(arg, varArgs));
    }

    @Override
    public void checkLegalityBeforeTypeCoercion() {
        String functionName = getName();
        if (this.children.isEmpty()) {
            throw new AnalysisException("The " + functionName + " function must have at least one param");
        }

        for (int i = 0; i < children.size(); i++) {
            if (!getArgumentType(i).isBooleanType()) {
                throw new AnalysisException("All params of " + functionName + " function must be boolean");
            }
        }
    }

    /**
     * withDistinctAndChildren.
     */
    @Override
    public Retention withDistinctAndChildren(boolean distinct, List<Expression> children) {
        Preconditions.checkArgument(children.size() >= 1);
        return new Retention(distinct, children.get(0),
                children.subList(1, children.size()).toArray(new Expression[0]));
    }

    @Override
    public <R, C> R accept(ExpressionVisitor<R, C> visitor, C context) {
        return visitor.visitRetention(this, context);
    }

    @Override
    public List<FunctionSignature> getSignatures() {
        return SIGNATURES;
    }
}
