package org.apache.torque.om;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.sql.Types;

/**
 * This class can be used as an ObjectKey to uniquely identify an
 * object within an application where the id consists
 * of a Boolean.
 *
 * @author <a href="mailto:jmcnally@apache.org">John McNally</a>
 * @version $Id: BooleanKey.java 1448414 2013-02-20 21:06:35Z tfischer $
 */
public class BooleanKey extends SimpleKey
{
    /**
     * Serial version
     */
    private static final long serialVersionUID = 5109588772086713341L;

    /**
     * Creates an SimpleKey whose internal representation will be
     * set later, through a set method
     */
    public BooleanKey()
    {
        // empty
    }

    /**
     * Creates a BooleanKey whose internal representation is a Boolean
     *
     * @param key the key value
     */
    public BooleanKey(Boolean key)
    {
        this.key = key;
    }

    /**
     * Creates a BooleanKey that is equivalent to key.
     *
     * @param key the key value
     */
    public BooleanKey(BooleanKey key)
    {
        if (key != null)
        {
            this.key = key.getValue();
        }
        else
        {
            this.key = null;
        }
    }

    /**
     * Sets the internal representation to a String.
     *
     * @param key the key value
     */
    public void setValue(String key)
    {
        if (key == null)
        {
            this.key = null;
        }
        else
        {
            this.key = Boolean.parseBoolean(key);
        }
    }

    /**
     * Sets the internal representation to a Boolean.
     *
     * @param key the key value
     */
    public void setValue(Boolean key)
    {
        this.key = key;
    }

    /**
     * Sets the internal representation to the same object used by key.
     *
     * @param key the key value
     */
    public void setValue(BooleanKey key)
    {
        if (key != null)
        {
            this.key = key.getValue();
        }
        else
        {
            this.key = null;
        }
    }

    /**
     * Access the underlying Boolean object.
     *
     * @return a <code>Boolean</code> value
     */
    public Boolean getBoolean()
    {
        return (Boolean) key;
    }

    /**
     * Returns the JDBC type of the key
     * as defined in <code>java.sql.Types</code>.
     *
     * @return <code>Types.BIT</code>.
     */
    public int getJdbcType()
    {
        return Types.BIT;
    }

    /**
     * Get a String representation of this key.
     *
     * @return a String representation of this key,
     *         or an empty String if the value is null.
     */
    @Override
    public String toString()
    {
        if (key != null)
        {
            return key.toString();
        }
        return "";
    }
}
