/*
 * Decompiled with CFR 0.152.
 */
package org.apache.brooklyn.core.mgmt.ha;

import com.google.common.annotations.Beta;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Maps;
import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.brooklyn.api.mgmt.ManagementContext;
import org.apache.brooklyn.api.mgmt.ha.ManagementNodeState;
import org.apache.brooklyn.api.mgmt.ha.ManagementNodeSyncRecord;
import org.apache.brooklyn.api.mgmt.ha.ManagementPlaneSyncRecord;
import org.apache.brooklyn.api.mgmt.ha.ManagementPlaneSyncRecordPersister;
import org.apache.brooklyn.core.mgmt.ha.dto.BasicManagementNodeSyncRecord;
import org.apache.brooklyn.core.mgmt.ha.dto.ManagementPlaneSyncRecordImpl;
import org.apache.brooklyn.core.mgmt.persist.MementoSerializer;
import org.apache.brooklyn.core.mgmt.persist.PersistenceObjectStore;
import org.apache.brooklyn.core.mgmt.persist.RetryingMementoSerializer;
import org.apache.brooklyn.core.mgmt.persist.StoreObjectAccessorLocking;
import org.apache.brooklyn.core.mgmt.persist.XmlMementoSerializer;
import org.apache.brooklyn.util.exceptions.Exceptions;
import org.apache.brooklyn.util.text.Strings;
import org.apache.brooklyn.util.time.Duration;
import org.apache.brooklyn.util.time.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Beta
public class ManagementPlaneSyncRecordPersisterToObjectStore
implements ManagementPlaneSyncRecordPersister {
    private static final Logger LOG = LoggerFactory.getLogger(ManagementPlaneSyncRecordPersisterToObjectStore.class);
    private static final Duration SHUTDOWN_TIMEOUT = Duration.TEN_SECONDS;
    private static final Duration SYNC_WRITE_TIMEOUT = Duration.TEN_SECONDS;
    public static final String NODES_SUB_PATH = "nodes";
    private final ConcurrentMap<String, PersistenceObjectStore.StoreObjectAccessorWithLock> nodeWriters = Maps.newConcurrentMap();
    private PersistenceObjectStore.StoreObjectAccessorWithLock planeIdReader;
    private PersistenceObjectStore.StoreObjectAccessorWithLock masterWriter;
    private PersistenceObjectStore.StoreObjectAccessorWithLock changeLogWriter;
    private ManagementContext mgmt;
    private final PersistenceObjectStore objectStore;
    private final MementoSerializer<Object> serializer;
    private static final int MAX_SERIALIZATION_ATTEMPTS = 5;
    private boolean started = false;
    private volatile boolean running = true;
    protected final AtomicLong checkpointLogCount = new AtomicLong();
    private static final int INITIAL_LOG_WRITES = 5;
    private boolean isStartup = false;
    @VisibleForTesting
    private boolean preferRemoteTimestampInMemento = false;

    public ManagementPlaneSyncRecordPersisterToObjectStore(ManagementContext mgmt, PersistenceObjectStore objectStore, ClassLoader classLoader) {
        this.mgmt = mgmt;
        this.objectStore = (PersistenceObjectStore)Preconditions.checkNotNull((Object)objectStore, (Object)"objectStore");
        XmlMementoSerializer rawSerializer = XmlMementoSerializer.XmlMementoSerializerBuilder.from(mgmt).withBrooklynDeserializingClassRenames().withClassLoader((ClassLoader)Preconditions.checkNotNull((Object)classLoader, (Object)"classLoader")).build();
        this.serializer = new RetryingMementoSerializer<Object>(rawSerializer, 5);
        objectStore.createSubPath(NODES_SUB_PATH);
        LOG.debug("ManagementPlaneMemento-persister will use store " + objectStore);
    }

    protected synchronized void init() {
        if (!this.started) {
            this.started = true;
            this.masterWriter = new StoreObjectAccessorLocking(this.objectStore.newAccessor("/master"));
            if (this.masterWriter.get() != null) {
                this.changeLogWriter = new StoreObjectAccessorLocking(this.objectStore.newAccessor("/change.log"));
            } else {
                this.masterWriter = new StoreObjectAccessorLocking(this.objectStore.newAccessor("master"));
                this.changeLogWriter = new StoreObjectAccessorLocking(this.objectStore.newAccessor("change.log"));
            }
            this.planeIdReader = new StoreObjectAccessorLocking(this.objectStore.newAccessor("planeId"));
        }
    }

    @VisibleForTesting
    public void preferRemoteTimestampInMemento() {
        this.preferRemoteTimestampInMemento = true;
    }

    public void stop() {
        this.running = false;
        try {
            for (PersistenceObjectStore.StoreObjectAccessorWithLock writer : this.nodeWriters.values()) {
                try {
                    writer.waitForCurrentWrites(SHUTDOWN_TIMEOUT);
                }
                catch (TimeoutException e) {
                    LOG.warn("Timeout during shutdown, waiting for write of " + writer + "; continuing");
                }
            }
            try {
                this.masterWriter.waitForCurrentWrites(SHUTDOWN_TIMEOUT);
            }
            catch (TimeoutException e) {
                LOG.warn("Timeout during shutdown, waiting for write of " + this.masterWriter + "; continuing");
            }
        }
        catch (InterruptedException e) {
            throw Exceptions.propagate((Throwable)e);
        }
    }

    public ManagementPlaneSyncRecord loadSyncRecord() throws IOException {
        return this.loadSyncRecord(Duration.ZERO);
    }

    public ManagementPlaneSyncRecord loadSyncRecord(Duration terminatedNodeDeletionTimeout) throws IOException {
        if (!this.running) {
            throw new IllegalStateException("Persister not running; cannot load memento from " + this.objectStore.getSummaryName());
        }
        this.init();
        if (LOG.isTraceEnabled()) {
            LOG.trace("Loading management-plane memento from {}", (Object)this.objectStore.getSummaryName());
        }
        Stopwatch stopwatch = Stopwatch.createStarted();
        ManagementPlaneSyncRecordImpl.Builder builder = ManagementPlaneSyncRecordImpl.builder();
        String masterNodeId = this.masterWriter.get();
        if (masterNodeId == null) {
            LOG.debug("No master-memento deserialized from file " + this.masterWriter + "; ignoring and continuing (normal on startup, should cause an error later in live operation)");
        } else {
            builder.masterNodeId(masterNodeId);
        }
        builder.planeId(Strings.emptyToNull((String)this.planeIdReader.get()));
        List<String> nodeFiles = this.objectStore.listContentsWithSubPath(NODES_SUB_PATH);
        LOG.trace("Loading nodes from {}; {} nodes.", new Object[]{this.objectStore.getSummaryName(), nodeFiles.size()});
        for (String nodeFile : nodeFiles) {
            Date now;
            Duration inactivityDuration;
            PersistenceObjectStore.StoreObjectAccessor objectAccessor = this.objectStore.newAccessor(nodeFile);
            String nodeContents = null;
            Exception problem = null;
            try {
                nodeContents = objectAccessor.get();
            }
            catch (Exception e) {
                Exceptions.propagateIfFatal((Throwable)e);
                problem = e;
            }
            if (problem != null || Strings.isBlank((CharSequence)nodeContents)) {
                if (objectAccessor.exists()) {
                    throw Exceptions.propagate((Throwable)new IllegalStateException("Node record " + nodeFile + " could not be read when " + this.mgmt.getManagementNodeId() + " was scanning", problem));
                }
                LOG.warn("Node record " + nodeFile + " went away while " + this.mgmt.getManagementNodeId() + " was scanning, ignoring (it has probably been terminated)");
                continue;
            }
            ManagementNodeSyncRecord memento = (ManagementNodeSyncRecord)this.serializer.fromString(nodeContents);
            if (memento == null) {
                throw Exceptions.propagate((Throwable)new IllegalStateException("Node record " + nodeFile + " could not be deserialized when " + this.mgmt.getManagementNodeId() + " was scanning: " + nodeContents, problem));
            }
            if (memento.getRemoteTimestamp() == null || !this.preferRemoteTimestampInMemento) {
                if (memento.getRemoteTimestamp() != null) {
                    LOG.debug("Ignoring remote timestamp in memento file (" + memento + "); looks like this data has been manually copied in");
                }
                Date lastModifiedDate = objectAccessor.getLastModifiedDate();
                ((BasicManagementNodeSyncRecord)memento).setRemoteTimestamp(lastModifiedDate != null ? Long.valueOf(lastModifiedDate.getTime()) : null);
            }
            if (terminatedNodeDeletionTimeout.compareTo(Duration.ZERO) == 1 && this.isStartup && memento.getStatus().name().equals("TERMINATED") && (inactivityDuration = new Duration((now = new Date()).getTime() - memento.getRemoteTimestamp(), TimeUnit.MILLISECONDS)).compareTo(terminatedNodeDeletionTimeout) == 1) {
                LOG.debug("Last modified date exceeds the provided threshold for: " + memento + "; node will be removed from persistence store.");
                try {
                    objectAccessor.delete();
                }
                catch (Exception e) {
                    LOG.debug("Exception: " + e + " while trying to remove node: " + memento + ". Progressing regardless...");
                }
                continue;
            }
            builder.node(memento);
        }
        if (this.isStartup) {
            this.isStartup = false;
        }
        if (LOG.isDebugEnabled()) {
            LOG.trace("Loaded management-plane memento; {} nodes, took {}", (Object)nodeFiles.size(), (Object)Time.makeTimeStringRounded((long)stopwatch.elapsed(TimeUnit.MILLISECONDS)));
        }
        return builder.build();
    }

    public void setIsStartup(boolean isStartup) {
        this.isStartup = isStartup;
    }

    public void delta(ManagementPlaneSyncRecordPersister.Delta delta) {
        if (!this.running) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Persister not running; ignoring checkpointed delta of manager-memento");
            }
            return;
        }
        this.init();
        Stopwatch stopwatch = Stopwatch.createStarted();
        if (LOG.isTraceEnabled()) {
            LOG.trace("Checkpointing delta of manager-memento; updating {}", (Object)delta);
        }
        for (ManagementNodeSyncRecord m : delta.getNodes()) {
            this.persist(m);
        }
        for (String id : delta.getRemovedNodeIds()) {
            this.deleteNode(id);
        }
        switch (delta.getMasterChange()) {
            case NO_CHANGE: {
                break;
            }
            case SET_MASTER: {
                this.persistMaster((String)Preconditions.checkNotNull((Object)delta.getNewMasterOrNull()), null);
                break;
            }
            case CLEAR_MASTER: {
                this.persistMaster("", delta.getExpectedMasterToClear());
                break;
            }
            default: {
                throw new IllegalStateException("Unknown state for master-change: " + delta.getMasterChange());
            }
        }
        if (LOG.isDebugEnabled() && this.shouldLogCheckpoint()) {
            LOG.debug("Checkpointed delta of manager-memento in " + Time.makeTimeStringRounded((Stopwatch)stopwatch) + ": " + delta);
        }
    }

    private void persistMaster(String nodeId, String optionalExpectedId) {
        String currentRemoteMaster;
        if (optionalExpectedId != null && (currentRemoteMaster = this.masterWriter.get()) != null && !currentRemoteMaster.trim().equals(optionalExpectedId.trim())) {
            LOG.warn("Master at server is " + (Strings.isBlank((CharSequence)currentRemoteMaster) ? "<none>" : currentRemoteMaster) + "; expected " + optionalExpectedId + " " + (Strings.isNonBlank((CharSequence)nodeId) ? "and would set as " + nodeId : "and would clear") + ", so not applying (yet)");
            return;
        }
        this.masterWriter.put(nodeId);
        try {
            this.masterWriter.waitForCurrentWrites(SYNC_WRITE_TIMEOUT);
        }
        catch (Exception e) {
            throw Exceptions.propagate((Throwable)e);
        }
        this.changeLogWriter.append(Time.makeDateString() + ": set master to " + nodeId + "\n");
        try {
            this.changeLogWriter.waitForCurrentWrites(SYNC_WRITE_TIMEOUT);
        }
        catch (Exception e) {
            throw Exceptions.propagate((Throwable)e);
        }
    }

    @VisibleForTesting
    public void waitForWritesCompleted(Duration timeout) throws InterruptedException, TimeoutException {
        for (PersistenceObjectStore.StoreObjectAccessorWithLock writer : this.nodeWriters.values()) {
            writer.waitForCurrentWrites(timeout);
        }
        this.masterWriter.waitForCurrentWrites(timeout);
    }

    public void checkpoint(ManagementPlaneSyncRecord record) {
        this.init();
        for (ManagementNodeSyncRecord node : record.getManagementNodes().values()) {
            if (ManagementNodeState.INITIALIZING.equals((Object)node.getStatus()) || node.getNodeId() == null) continue;
            this.persist(node);
        }
    }

    private void persist(ManagementNodeSyncRecord node) {
        PersistenceObjectStore.StoreObjectAccessorWithLock writer = this.getOrCreateNodeWriter(node.getNodeId());
        boolean fileExists = writer.exists();
        writer.put(this.serializer.toString(node));
        try {
            writer.waitForCurrentWrites(SYNC_WRITE_TIMEOUT);
        }
        catch (Exception e) {
            throw Exceptions.propagate((Throwable)e);
        }
        if (!fileExists) {
            this.changeLogWriter.append(Time.makeDateString() + ": created node " + node.getNodeId() + "\n");
        }
        if (node.getStatus() == ManagementNodeState.TERMINATED || node.getStatus() == ManagementNodeState.FAILED) {
            this.changeLogWriter.append(Time.makeDateString() + ": set node " + node.getNodeId() + " status to " + node.getStatus() + "\n");
        }
    }

    private void deleteNode(String nodeId) {
        this.getOrCreateNodeWriter(nodeId).delete();
        this.changeLogWriter.append(Time.makeDateString() + ": deleted node " + nodeId + "\n");
    }

    private PersistenceObjectStore.StoreObjectAccessorWithLock getOrCreateNodeWriter(String nodeId) {
        PersistenceObjectStore.StoreObjectAccessorWithLock writer = (PersistenceObjectStore.StoreObjectAccessorWithLock)this.nodeWriters.get(nodeId);
        if (writer == null) {
            this.nodeWriters.putIfAbsent(nodeId, new StoreObjectAccessorLocking(this.objectStore.newAccessor("nodes/" + nodeId)));
            writer = (PersistenceObjectStore.StoreObjectAccessorWithLock)this.nodeWriters.get(nodeId);
        }
        return writer;
    }

    protected boolean shouldLogCheckpoint() {
        long logCount = this.checkpointLogCount.incrementAndGet();
        return logCount < 5L || logCount % 1000L == 0L;
    }
}

