/*  
    CipherSaber, Copyright (c) 2002-2004 Kalle Raisanen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#ifdef HAVE_SHA
#include "sha/sha2.c"
#endif

#ifdef UNIX
#include <unistd.h>
#include <termio.h>
#endif

/* PRNGs (c) George Marsaglia */
#define znew  (zold=36969*(zold&0xffff)+(zold>>16))
#define wnew  (wold=18000*(wold&0xffff)+(wold>>16))
#define MWC   ((znew<<16)+wnew)
#define SHR3  (jsr^=(jsr<<17), jsr^=(jsr>>13), jsr^=(jsr<<5))
#define CONG  (jcong=69069*jcong+1234567)
#define KISS  ((MWC^CONG)+SHR3)
unsigned long zold, wold, jsr, jcong;

#define UINT_SIZE (sizeof(unsigned int) * 8)

#define getrndbyte() (KISS>>24)

#define VERSTAG  "CipherSaber 0.7 (2004-04-17), Copyright (c) 2002-2004 Kalle Raisanen.\n"

#define BUFFSIZE 5120
#define IV_SIZE    10
#define STATELEN  256
#define SHA_SIZE   64
#define MCHAR     246
#define MAX_PATH  256

const char COPYMESS[] = 
   {
   VERSTAG 
   "\nThis program is free software; you can redistribute it and/or modify\n"
   "it under the terms of the GNU General Public License as published by\n"
   "the Free Software Foundation; either version 2 of the License, or\n"
   "(at your option) any later version.\n\n"
   "This program is distributed in the hope that it will be useful,\n"
   "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
   "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
   "GNU General Public License for more details.\n\n"
   "You should have received a copy of the GNU General Public License\n"
   "along with this program; if not, write to the Free Software\n"
   "Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n\n\n"
   "SHA-512 Code, Copyright (c) 2000-2001, Aaron D. Gifford. All rights reserved.\n\n"
   "Redistribution and use in source and binary forms, with or without\n"
   "modification, are permitted provided that the following conditions\n"
   "are met:\n\n"
   "   1. Redistributions of source code must retain the above copyright\n"
   "      notice, this list of conditions and the following disclaimer.\n"
   "   2. Redistributions in binary form must reproduce the above copyright\n"
   "      notice, this list of conditions and the following disclaimer in the\n"
   "      documentation and/or other materials provided with the distribution.\n"
   "   3. Neither the name of the copyright holder nor the names of contributors\n"
   "      may be used to endorse or promote products derived from this software\n"
   "      without specific prior written permission.\n\n"
   "THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTOR(S) ``AS IS'' AND\n"
   "ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE\n"
   "IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE\n"
   "ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTOR(S) BE LIABLE\n"
   "FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL\n"
   "DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS\n"
   "OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)\n"
   "HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT\n"
   "LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY\n"
   "OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF\n"
   "SUCH DAMAGE.\n"   
   };
const char HELPMESS[] =
   {
   VERSTAG
   "CS encrypts/decrypts files using the RC-4 algorithm, with a 10-byte (pseudo-)random\n" 
   "initializaton vector (IV), optionally hashing the IV and passphrase using SHA-512.\n\n"
   "Usage: cs [-OPTION]... <FILE>...\n\n"
   "Where OPTION is one or more of:\n\n"
   "  -d, --decrypt          Decrypt file, if not specified, the file will be encrypted.\n"
   "  -e, --add-entropy      Add entropy. You get to type a bit, and CS gathers entropy from the\n"
   "                         delays between your keystrokes. Only works when encrypting.\n"
   "  -i, --ignore-errors    Ignore errors.\n"
   "  -o, --to-stdout        Output to standard output. Only works when decrypting.\n"
   "  -p, --same-passphrase  Use same passphrase for all files.\n"
   "  -q, --quiet            Quiet mode, no error messages.\n"
#ifdef HAVE_SHA
   "  -s, --sha              Use SHA-512 hash on passphrase and IV.\n"
#endif
   "      --help             Output this message, and exit.\n"
   "      --copyright        Output copyright info and exit.\n"
   "      --version          Output version info and exit.\n\n"
   "Report any bugs to <bugs@aros-stuff.tk>.\n"
   };
const char NOARGS[] =
   {
   VERSTAG
   "\nYou must specify a filename, try 'cs --help' for more information.\n"
   };

#ifdef UNIX
#	include <termios.h>

	static struct termios stored_settings;

	void raw_mode(int raw)
	{
		struct termios new_settings;
	
		if(raw)
			{
			tcgetattr(0, &stored_settings);
			new_settings = stored_settings;
			new_settings.c_lflag &= (~ICANON);
			new_settings.c_cc[VTIME] = 0;
			new_settings.c_cc[VMIN]  = 1;

			tcsetattr(0, TCSANOW, &new_settings);
			}
		else
			tcsetattr(0, TCSANOW, &stored_settings);
	}
#else
#	ifdef AROS
#		include <proto/dos.h>
#		define raw_mode(r) SetMode(Input(), r)
#	else
#		warning "Default (not AROS, not Unix) code hasn't been tested, and may not work."
#		define raw_mode(x) fprintf(stderr, "raw_mode('%s'): not yet implemented.\n", x?"on":"off")
#	endif /* AROS */
	char *getpass(char *prompt)
	{
		int i, c;
   	char *buff;

	   if(!(buff = (char *)malloc(MCHAR + 1)))
			{
			fprintf(stderr, "cs: malloc() of %d bytes for input buffer failed!\n", MCHAR);
			exit(10);
			}

	   printf(prompt);

		raw_mode(1);

		for(i = 0; ((c = getchar()) != '\n') && (c != '\r') && (i < MCHAR); i++)
			buff[i] = c;
		buff[i] = 0;
		raw_mode(0);
		printf("\n");
   	return buff;
	}
#endif /* UNIX */


int cs(char *filename, char *passwd, int dcrypt, int sha, int std)
{
   FILE *in, *out;
   int i = 0, index = 0, keylen = 0, buflen = 0;
   char *end;
   unsigned char state[STATELEN], key[STATELEN] = "", a = 0, b = 0, temp = 0,
                 buffer[BUFFSIZE], init_vec[IV_SIZE], outname[MAX_PATH];
#ifdef HAVE_SHA
   unsigned char sha_out[SHA_SIZE+1];
   SHA512_CTX shac;
#endif

   if(!(in  = fopen(filename, "rb")))
      return 0;   
   
   strcpy(outname, filename);
   strcat(outname, "\x01"); /* so we only remove the last .cs */
   if(!dcrypt)      
      {
      outname[strlen(outname)-1] = 0;
      strcat(outname, ".cs");
      }
   else if( (end = strstr(outname, ".cs\x01"))  || (end = strstr(outname, ".CS\x01"))  ||
            (end = strstr(outname, ".cs1\x01")) || (end = strstr(outname, ".CS1\x01")) ||
            (end = strstr(outname, ".cs2\x01")) || (end = strstr(outname, ".CS2\x01")) )
      end[0] = 0;
   
   strcpy(key, passwd);
   if(!(out = (std) ? stdout : fopen(outname, "wb")))
      {
      fclose(in);
      return 0;
      }

   /* initiate vector */
   if(dcrypt)
      fread(init_vec, 1, IV_SIZE, in);
   else
      {
      for(i = 0; i < IV_SIZE; i++)
         init_vec[i] ^= getrndbyte();
      fwrite(init_vec, 1, IV_SIZE, out);
      }
   strcat(key, init_vec);
#ifdef HAVE_SHA
   if(sha)
      {
      SHA512_Init(&shac);
      SHA512_Update(&shac, key, strlen(key));
      SHA512_Final(sha_out, &shac);
      sha_out[SHA_SIZE] = 0;
      memset(key, 0, STATELEN);
      strcpy(key, sha_out);
      memset(sha_out, 0, SHA_SIZE);
      }
#endif
   keylen = strlen(key);

   /* prepare state */
   for(i = 0; i < STATELEN; i++)
      state[i] = i;
   for(i = 0; i < STATELEN; i++)
      {
      b = (b + key[a] + state[i]) % STATELEN;
      temp = state[i];
      state[i] = state[b];
      state[b] = temp;
      a = (a + 1) % keylen;
      }
   memset(key, 0, STATELEN);
   /* encrypt/decrypt */
   a = b = keylen = 0;
   while(buflen = fread(buffer, 1, BUFFSIZE, in))
      {
      for(i = 0; i < buflen; i++)
         {
         a = (a + 1) % STATELEN;
         b = (b + state[a]) % STATELEN;
         temp = state[a];
         state[a] = state[b];
         state[b] = temp;
         index = (state[a] + state[b]) % STATELEN;
         buffer[i] ^= state[index];
         }
      fwrite(buffer, 1, buflen, out);
      }
   fclose(in);
   if(!std)
      fclose(out);
   memset(state, 0, STATELEN);
   memset(buffer, 0, BUFFSIZE);

   return 1;
}

#define randomize(x) x ^= ((unsigned long)time((time_t *)NULL) % 2) << (i % UINT_SIZE)

void entropize()
{
   int i;
   printf("Type a bit (At least %d chars needed), end with ENTER. \n", UINT_SIZE*4); 
 
 	raw_mode(1);

	printf("\r%03d\t", 0); 
 
   for(i = 0; getchar() != '\n'; i++)
      {
      if(i < UINT_SIZE)
			randomize(zold);
		else if(i < (UINT_SIZE*2))
			randomize(wold);
		else if(i < (UINT_SIZE*3))
			randomize(jcong);
		else if(i < (UINT_SIZE*4))
			randomize(jsr);
		printf("\r%03d\t", i); 
      }
     
  raw_mode(0);
  printf("%d chars received\n\n", i);
}

int main(int argc, char *argv[])
{
   int dc = 0, sh = 0, st = 0, pw = 0, mp = 0, 
       pg = 0, en = 0, ig = 0, si = 0, pf = 0; 
   int uo = 0, i, ff = 0;
   char passwd[MCHAR], *buff;

   for(i = 1; i < argc; i++)
      {
      buff = argv[i];
      if(!strcmp(buff, "--copyright"))
         {
         printf(COPYMESS); return 0;
         }
      else if(!strcmp(buff, "--decrypt"))
         dc = 1;
      else if(!strcmp(buff, "--add-entropy"))
         en = 1;
      else if(!strcmp(buff, "--help"))
         {
         printf(HELPMESS); return 0;
         }
      else if(!strcmp(buff, "--ignore-errors"))
         ig = 1;
      else if(!strcmp(buff, "--to-stdout"))
         st = 1;
      else if(!strcmp(buff, "--same-passphrase"))
         mp = 1;
      else if(!strcmp(buff, "--quiet"))
         si = 1;
#ifdef HAVE_SHA
      else if(!strcmp(buff, "--sha"))
         sh = 1;
#endif
      else if(!strcmp(buff, "--version"))
         {
         printf(VERSTAG); return 0;
         }
      else if(*buff++ == '-')
         {
         while(*buff)
            {
            switch(*buff++)
               {
               case 'd': dc = 1; break;
               case 'e': en = 1; break;
               case 'i': ig = 1; break;
               case 'o': st = 1; break;
               case 'p': mp = 1; break;
               case 'q': si = 1; break;
#ifdef HAVE_SHA
               case 's': sh = 1; break;
#endif
               default:  uo = *buff - 1; break;
               }
            }
         }
      else
         ff = 1;
      }

   if(!ff || argc < 2)
      {
      fprintf(stderr, NOARGS);
      return 5;
      }

   if(uo)
      {
      if(!si) fprintf(stderr, "%s: unknown option %c!\n", argv[0], uo);
      if(!ig) return 5;
      }
   else if(st && !dc)
      {
      if(!si) fprintf(stderr, "%s: -o can only be used when decrypting!\n", argv[0]);
      if(!ig) return 5;
      }
   else if(en && dc)
      {
      if(!si) fprintf(stderr, "%s: -e can only be used when encrypting!\n", argv[0]);
      if(!ig) return 5;
      }
   zold = wold = jsr = jcong = (unsigned long)time((time_t *)NULL);

   if(en) entropize();
   
   for(i = 1; i < argc || !ff; i++)
      {
      if(strncmp(argv[i], "-", 1))
         {
         if(!pg)
            strcpy(passwd, getpass("Passphrase: "));

         if(!cs(argv[i], passwd, dc, sh, st))
            {
            if(!si) fprintf(stderr, "%s: %scryption of %s failed!\n", argv[i], (dc) ? "de" : "en", argv[i]);
            if(!ig) return 5;
            }
         if(!mp)
            memset(passwd, 0, MCHAR);          
         else
            pg = 1;
         }
      }
   return 0;   
}
