/***************************************************************************/
/*                          Amiga floppy reader                            */
/*                        (c) 1997, Marco Veneri                           */
/*                                 v1.0f                                   */
/***************************************************************************/

// Compiler: Borland CPP v2.0 (anyone else should work fine also)
// Tab size: 4

// NOTE: I do NOT assume any responsibility for the consequences of use
// 	of this program and circuit nor for any problem, trouble or
//	hardware damage which may result from its use.
//
// For any question, suggestion or doubt please leave a mail at:
// E-MAIL : marco_veneri@hotmail.com

/*
** NOTE: How to set LPT DATA port direction.
**	 On my computer bit 5 of the LPT CONTROL byte seems to be used
**	 to control DATA port flow direction. The problem is that
**	 I have not found any document reporting this bit (it looks
**	 like if this bit is unused).
*/

/*
** ABOUT PIN USAGE:
** 	Some LPT port pin has more than one use in my interface circuit.
**	This is not a problem since lines using same pin have to be set
**	or reset only before a latching pulse.
*/

#include    <stdio.h>
#include    <stdlib.h>
#include    <string.h>
#include    <ctype.h>
#include    <dos.h>
#include    <conio.h>
#include    <io.h>

/*============================= DEFINES ===================================*/

#define DATA_PORT       (LPTBaseAddress)
#define STATUS_PORT     (DATA_PORT + 1)
#define CONTROL_PORT    (DATA_PORT + 2)

/*
** signals tagged with 'NOT' have inverted logic value as the
** corresponding pin on the LPT. Note that some pin of the LPT
** port control more than just one function.
*/

#define B_DATA_READY    (1<<6)  /* STATUS PORT          */
#define B_FLOPPY_READY  (1<<7)  /* STATUS PORT   (NOT)  */
#define B_TRACK_ZERO    (1<<5)  /* STATUS PORT          */

#define B_HEAD_STEP     (1<<0)  /* CONTROL PORT  (NOT)  */
#define B_HEAD_DIR      (1<<1)  /* CONTROL PORT  (NOT)  */
#define B_SIDE          (1<<1)  /* CONTROL PORT  (NOT)  */
#define B_ENABLE_DATA   (1<<2)  /* CONTROL PORT         */
#define B_MOTOR         (1<<2)  /* CONTROL PORT         */
#define B_SELECT        (1<<3)  /* CONTROL_PORT  (NOT)  */

#define LPT_INPUT_MODE_BIT (1<<5)   /* set DATA port as an input port */

#define SUCCESS  		1
#define FAILURE         0
#define TRUE            1
#define FALSE           0

/* exactly: (512 + 32) * 2 * 11 = 11968
**		512 = data, 32 = header stuff.
**
**  	Header structure:
**
**	- 2 bytes: (00h,00h -> AAAAh,AAAAh in MFM)
**	- 2 bytes: (A1h,A1h -> 4489h,4489h in MFM)	standard sync word
**	- 4 bytes: FFh, TrackNumber, SectorNumber, SectorsToEndOfTrack
**	-16 bytes: OS Recovery data. (Someone knows they're usage ????)
**	- 4 bytes: Header checksum longword
**	- 4 bytes: Data checksum longword
*/

#define BYTES_PER_TRACK_READ    13800
#define	NOT_FOUND_MARK	   		20

#define	RETRY_BEFORE_ASK		10

/*=========================== PROTOTYPES ==================================*/

void MotorOn(void);
void MotorOff(void);

void TrackIn(void);
void TrackOut(void);
void TrackSeek(int trk);
void TrackZero(void);

void selectSide(int side);

void ImportTrack(void);
void ReadTrack(void);
void saveTrack(FILE *file);
int  processTrack(void);

unsigned int  GetMFMData(unsigned int bytePos, unsigned int bitPos);
unsigned long GetMFMLong(unsigned int offset);

int CheckForSyncPattern(unsigned int bytePos, unsigned int bitPos);
unsigned int GotoNextSyncPattern(unsigned int startByte);

unsigned long DecodeLongword(unsigned int offset);
int  DecodeSectorHeader(void);
void DecodeSectorData(int sector);

unsigned long CalcSectorHeaderCHK(void);
unsigned long CalcSectorDataCHK(void);

void beep(void);
void gaugeClear(void);
void gaugePulse(void);

void MakeScreen(void);
void PrintStatus(int sector, int track, int code);

void ReadDisk(FILE *file);
int  AskForFilename(void);
void AskForLPTPortAddress(void);
int AskForRetry(void);
void PrintMessage(char *msg);

/*=========================== Sector Structure ============================*/

struct Sector
{
	unsigned char format_type;   	/* should always be 0xFF 		 	*/
	unsigned char track;
	unsigned char sector;
	unsigned char offset;		/* = NOT_FOUND_MARK if not found	*/
	long data_chk;
	unsigned char data[512];
};

/*=============================== GLOBALS =================================*/

int LPTBaseAddress = 0x378;

int     Track;	 	/* the actual track  			  	*/
int 	Side;	       	/* used by ProcessTrack to print infos 		*/
unsigned char buffer[BYTES_PER_TRACK_READ + 1];	   /* full track buffer */
unsigned char control;	/* always equal to the LPT control port 	*/
unsigned char status;   /* always equal to the LPT status port 		*/

unsigned int bPosByte;	/* offset in the track buffer 			*/
unsigned int bPosBit;   /* actual bit alignment				*/

/* magic sync bit pattern preceding each sector on track */
unsigned int syncPattern[4] = { 0xAAAA, 0xAAAA, 0x4489, 0x4489 };

/* here is where we put decoded sector data */
struct Sector sector[11];

int gaugePos;
int missing;		/* current track missing sector number 	*/

char filename[13] = "********.adf";

//===========================================================================
//				selectSide()
//===========================================================================

void selectSide(int side)
{
	if(!side)
	{	// select side 0
		control &= ~B_SIDE;
		Side=0;
		outportb(CONTROL_PORT, control);
	}
	else
	{
		// select side 1
		control |= B_SIDE;
		Side = 1;
		outportb(CONTROL_PORT, control);
	}
}

//===========================================================================
//			      ImportTrack()
//===========================================================================

/*
** NOTE: this function was originally inplemented with asm code.Since we
** 	have a 74LS374 latch gate data must be read within 4*2 microseconds
**	since latch front-edge pulse; I think this time is long enough for
**	any 'good' computer even using plain C code.
*/

void ImportTrack(void)
{
	register int i;

	disable();

	for(i=0; i<BYTES_PER_TRACK_READ; i++)
	{
		// wait for the rising edge of the latch pulse
		while((inp(STATUS_PORT) & 0x40));
		while(!(inp(STATUS_PORT) & 0x40));
		// read the data port
		buffer[i] = ~inp(DATA_PORT);
	}

	enable();
}

//===========================================================================
//			       MotorOn()
//===========================================================================

void MotorOn(void)
{
	/* set motor bit low */
	control = 0;
	outportb(CONTROL_PORT, control);

	/* deselect drive */
	control &= ~(B_SELECT);
	outportb(CONTROL_PORT, control);

	/* select drive */
	control |= (B_SELECT);
	outportb(CONTROL_PORT, control);

	delay(500);

//	/* wait while floppy is speeding up */
//	while(!(inportb(STATUS_PORT) & B_FLOPPY_READY));
}

//===========================================================================
//				 MotorOff()
//===========================================================================

void MotorOff(void)
{
	/* reset motor bit */
	control |= B_MOTOR;
	outportb(CONTROL_PORT, control);

	/* deselect drive */
	control &= ~B_SELECT;
	outportb(CONTROL_PORT, control);

	/* select drive */
	control |= B_SELECT;
	outportb(CONTROL_PORT, control);
}

//===========================================================================
//       			 TrackIn()
//===========================================================================

void TrackIn(void)
{
	if(Track >= 79)
		return;

	/* reset DIRECTION bit */
	control |= B_HEAD_DIR;
	outportb(CONTROL_PORT, control);

	/* send a low pulse on STEP pin */
	control |= B_HEAD_STEP;
	outportb(CONTROL_PORT, control);
	control &= ~B_HEAD_STEP;
	outportb(CONTROL_PORT, control);

	/* wait a little */
	delay(5);

	Track++;
}
//===========================================================================
//			         TrackOut()
//===========================================================================

void TrackOut(void)
{
	if(Track <= 0)
		return;

	/* set DIRECTION bit */
	control &= ~B_HEAD_DIR;
	outportb(CONTROL_PORT, control);

	/* send a low pulse on STEP pin */
	control |= B_HEAD_STEP;
	outportb(CONTROL_PORT, control);
	control &= ~B_HEAD_STEP;
	outportb(CONTROL_PORT, control);

	/* wait a little */
	delay(5);

	Track--;
}

//===========================================================================
//			       TrackSeek()
//===========================================================================

void TrackSeek(int trk)
{
	while(Track < trk)
		TrackIn();

	while(Track > trk)
		TrackOut();
}

//===========================================================================
//				TrackZero()
//===========================================================================

void TrackZero(void)
{
	/* set DIRECTION bit as OUT */
//	control &= ~B_HEAD_DIR;
	control = B_SELECT | B_MOTOR;
	outportb(CONTROL_PORT, control);

	status = inportb(STATUS_PORT);

	while(status & B_TRACK_ZERO)
	{
		/* send a low pulse on STEP pin */
		control |= B_HEAD_STEP;
		outportb(CONTROL_PORT, control);
		control &= ~B_HEAD_STEP;
		outportb(CONTROL_PORT, control);

		/* wait a little */
		delay(5);

		status = inportb(STATUS_PORT);
	}

	Track = 0;
}

//===========================================================================
//			      GetMFMData()
//===========================================================================

/*
** This function extract from the bit-stream (data in the track
** buffer) a 16 bit integer whose position (in bits) is:
** bytePos*8 + (7-bitPos) (so 7 is byte-aligned).
*/

unsigned int GetMFMData(unsigned int bytePos, unsigned int bitPos)
{
	unsigned int data_A,
			data_B,
			data_C;

	data_A = ((unsigned int)(buffer[bytePos])) << (8 + 7 - bitPos);
	data_B = ((unsigned int)(buffer[bytePos+1])) << (7 - bitPos);
	data_C = ((unsigned int)(buffer[bytePos+2])) >> bitPos + 1;

	return(data_A | data_B | data_C);
}

//===========================================================================
//			       GetMFMLong()
//===========================================================================

/*
** same as GetMFMData, but extract a 32-bit value
*/

unsigned long GetMFMLong(unsigned int offset)
{
	unsigned long low, high;

	high = GetMFMData(bPosByte + offset + 0 , bPosBit);
	low = GetMFMData(bPosByte + offset + 2, bPosBit);

	 return(((high<<16) | low) & 0x55555555);
}

//===========================================================================
//			      CheckForSyncPattern()
//===========================================================================

/*
** This function scan the next 4 bytes to see if we have a
** magic sync pattern value.
*/

int CheckForSyncPattern(unsigned int bytePos, unsigned int bitPos)
{
	if (GetMFMData(bytePos+4, bitPos) != syncPattern[2]) return(FAILURE);
	if (GetMFMData(bytePos+6, bitPos) != syncPattern[3]) return(FAILURE);

	return(SUCCESS);
}

//===========================================================================
//			   GotoNextSyncPattern()
//===========================================================================

/*
** RETURNS : (unsigned int) position of the next Sync
**    	pattern. If no sync pattern is found return 64000
**	Moreover global var bPosBit is properly set.
** NOTE: this function scan all the buffer from startByte to the end using
**  	ALL eight possible bPosBit value searchin for a sync-pattern.Read the
**	note at the end of this procedure to speed up disk reading process.
*/

unsigned int GotoNextSyncPattern(unsigned int startByte)
{
	unsigned int i, pos;
	unsigned int nearestByte=64000;

	for(i=0; i<8; i++)
	{
		pos = startByte;

		// scan until sync-pattern or end of buffer is found
		while(!CheckForSyncPattern(pos, i))
		{
			pos++;

			if(pos >= BYTES_PER_TRACK_READ)
				break;
		}

		// if a sync-pattern was found....
		if(CheckForSyncPattern(pos, i))
		{
			if(pos < nearestByte)
			{
				nearestByte = pos;
				bPosBit = i;
	/*
	** NOTE: since we have found a sync-pattern we can return
	**		immediately. This would speed up disk reading
	**		process but, maybe, some hard-to-read disk would
	**		require much more retries
	*/
			}
		}
	}

	return(nearestByte);
}

//===========================================================================
//		             DecodeLongword()
//===========================================================================

/*
** This function read 4 MFM coded 16-bit value (i.e. 4 uncoded bytes) from
** the track buffer and return them as an unsigned longword.
*/

unsigned long DecodeLongword(unsigned int offset)
{
	unsigned int tmp[4];
	unsigned long decoded;

	/* get MFM data (4 bytes uncoded = 4*2 bytes MFM coded */
	tmp[0] = GetMFMData(bPosByte + offset + 0, bPosBit);
	tmp[1] = GetMFMData(bPosByte + offset + 2, bPosBit);
	tmp[2] = GetMFMData(bPosByte + offset + 4, bPosBit);
	tmp[3] = GetMFMData(bPosByte + offset + 6, bPosBit);

	/* get only data bits, reset all MFM added bits */
	tmp[0] &= 0x5555;
	tmp[1] &= 0x5555;
	tmp[2] &= 0x5555;
	tmp[3] &= 0x5555;

	/* this are the even bits */
	tmp[0] = (tmp[0]<<1);
	tmp[1] = (tmp[1]<<1);

	/* merge with odd bits */
	tmp[0] |= tmp[2];
	tmp[1] |= tmp[3];

	/* make a longword */
	decoded = tmp[0];
	decoded = ((decoded<<16) | tmp[1]);
	return(decoded);
}

//===========================================================================
//			  CalcSectorHeaderCHK()
//===========================================================================

unsigned long CalcSectorHeaderCHK(void)
{
	int i;
	unsigned long odd, even;
	unsigned long chk = 0L;

	odd = GetMFMLong(8);
	even = GetMFMLong(12);
	chk = odd ^ even;

	for(i=0; i<4; i++)
	{
		odd = GetMFMLong(16 + (i<<2));
		even = GetMFMLong(32 + (i<<2));
		chk ^= odd ^ even;
	}

	return(chk);
}

//===========================================================================
//			     CalcSectorDataCHK()
//===========================================================================

unsigned long CalcSectorDataCHK(void)
{
	int i;
	unsigned long odd, even;
	unsigned long chk = 0L;

	for(i=0; i<128; i++)
	{
		odd = GetMFMLong(64 + (i<<2));
		even = GetMFMLong(576 + (i<<2));
		chk ^= odd ^ even;
	}

	return(chk);
}

//===========================================================================
//			     DecodeSectorHeader()
//===========================================================================

/*
** PURPOSE:
**		- extract FORMAT_TYPE
**		- extract TRACK_NUMBER
**		- extract SECTOR_NUMBER
**		- extract OFFSET_NUMBER
**		- extract DATA_CHECKSUM
**		- check   HEADER_CHECKSUM
**		- return decoded SECTOR_NUMBER
*/

int DecodeSectorHeader(void)
{
	unsigned long lw;
	int sec;

	lw = DecodeLongword(8);
	sec = (unsigned char)((lw>>8) & 0x000000ff);

	// if we already have this sector then go back
	if(sector[sec].offset != NOT_FOUND_MARK)
		return(NOT_FOUND_MARK);

	// if header checksum is not correct return NOT_FOUND flag
	if(CalcSectorHeaderCHK() != DecodeLongword(48))
	{
		PrintStatus(sec, Track, -3);
		return(NOT_FOUND_MARK);
	}

	sector[sec].offset = (unsigned char)(lw & 0x000000ff);
	sector[sec].sector = (unsigned char)((lw>>8) & 0x000000ff);
	sector[sec].track = (unsigned char)((lw>>16) & 0x000000ff);
	sector[sec].format_type = (unsigned char)((lw>>24) & 0x000000ff);

	sector[sec].data_chk = DecodeLongword(56);

	return(sec);
}

//===========================================================================
//			      DecodeSectorData()
//===========================================================================

void DecodeSectorData(int sec)
{
	unsigned int i;
	unsigned int data;

	/* read all the even bits */
	for(i=0; i<256; i++)
	{
		data = GetMFMData(bPosByte+(i*2) + 64, bPosBit);
		data &= 0x5555;			/* reset even bits */
		sector[sec].data[(i*2)+0] = (unsigned char)(data>>7);
		sector[sec].data[(i*2)+1] = (unsigned char)(data<<1);
	}

	/* make an OR with all odd bits */
	for(i=0; i<256; i++)
	{
		data = GetMFMData(bPosByte + (i*2) + 64 + 512, bPosBit);
		data &= 0x5555;         /* reset even bits */
		sector[sec].data[(i*2)+0] |= (unsigned char)(data>>8);
		sector[sec].data[(i*2)+1] |= (unsigned char)(data);
	}
}

//===========================================================================
//			       saveTrack()
//===========================================================================

void saveTrack(FILE *file)
{
	int sec;

	for(sec=0; sec<11; sec++)
			fwrite(sector[sec].data, 512, 1, file);
}

//===========================================================================
//			       ReadTrack()
//===========================================================================

void ReadTrack(void)
{
	int i, done = FALSE;
	int retry=0;

	/* marks all sectors as not found */
	for(i=0; i<11; i++)
		sector[i].offset = NOT_FOUND_MARK;

	missing = 11;

	while(!done)
	{
		ImportTrack();
		done = TRUE;

		if(!processTrack())
		{
			beep();
			done = FALSE;
			retry++;
			if(retry == RETRY_BEFORE_ASK)
			{
				retry=0;
				if(!AskForRetry())
					done = TRUE;
			}
		}
	}
}

//===========================================================================
//					AskForRetry()
//===========================================================================

int AskForRetry(void)
{
	char c;

	beep();
	textcolor(YELLOW);textbackground(BLACK);
	gotoxy(25, 23);cprintf("Retry [R], Ignore [I] or Quit [q] ?");

	c = getch();
	textcolor(YELLOW);textbackground(BLACK);
	gotoxy(25, 23);cprintf("                                   ");


	if((c == 'i') || (c == 'I'))
		return(FALSE);
	else if((c == 'q') || (c == 'Q'))
		exit(1);
	else
		return(TRUE);
}

//===========================================================================
// 			     processTrack()
//===========================================================================

int processTrack(void)
{
	int i, sec, badFlag;
	unsigned int k;
	unsigned long dataChk;

	// position to start of track-buffer
	bPosByte = 0;
	badFlag=0;

	gaugeClear();

	k = GotoNextSyncPattern(bPosByte);
	if(k==64000)
		return(0);

	bPosByte = k;

	/* scan buffer to search start offset of each sector */
	while(missing & (bPosByte < BYTES_PER_TRACK_READ))
	{
			gaugePulse();

			/* process sector header */
			sec = DecodeSectorHeader();
			/* now we know sector number */

			if(sec != NOT_FOUND_MARK)
			{
				DecodeSectorData(sec);
				dataChk = CalcSectorDataCHK();

				if(dataChk != sector[sec].data_chk)
				{   // checksum error occurred
					sector[sec].offset = NOT_FOUND_MARK;
					PrintStatus(sec, Track, -2);
				}
				else
					PrintStatus(sec, Track, 0);
			}

			if(CheckForSyncPattern(bPosByte + 0x440, bPosBit))
				bPosByte += 0x440;
			else
				bPosByte = GotoNextSyncPattern(bPosByte+1);
	}

	// check if we have found all 11 sectors
	for(k=0; k<11; k++)
		if(sector[k].offset == NOT_FOUND_MARK)
			badFlag = 1;

	return(badFlag?0:1);
}

//===========================================================================
//				beep()
//===========================================================================

void beep(void)
{
	sound(500);
	delay(100);
	nosound();
}

//===========================================================================
//			       MakeScreen()
//===========================================================================

void MakeScreen(void)
{
	int sec, trk;

	textbackground(BLACK);
	clrscr();

	// print title block
	textbackground(BLUE);
	textcolor(YELLOW);gotoxy(22, 1);
	cputs("         AMIGA FLOPPY READER           ");
	textcolor(WHITE);gotoxy(22, 2);
	cputs("           by  Marco Veneri            ");

	// print sector map
	for(sec=0; sec<11; sec++)
		for(trk=0; trk<80; trk++)
			PrintStatus(sec, trk, -1);

	// print gauge block
	gaugeClear();

	// print filename block
	gotoxy(64, 20);textbackground(LIGHTGRAY);textcolor(RED);
	cprintf("    FILENAME    ");
	gotoxy(64, 21); cprintf("                ");
	gotoxy(72 - (strlen(filename)/2), 21);textcolor(BLACK);
	cprintf("%s", filename);
}

//===========================================================================
//			     PrintStatus()
//===========================================================================

/*
**	-1 = '|' : still to be read
**    	 0 = '#' : sector read and verified
**	-2 = 'D' : sector read but have a checksum error
**	-3 = 'H' : sector read but have header checksum
*/

void PrintStatus(int sector, int track, int code)
{
	unsigned short *screen = (unsigned short*)MK_FP(0xb800, 0x0280);

	if(code == -1)
		code = '|' | 0x0700;
	else if(code == -2)
		code = 'D' | 0x0500;
	else if(code == -3)
		code = 'H' | 0x0500;
	else if(code == 0)
	{
		if(Side == 0)
			code = '#' | 0x0B00;
		else
			code = '#' | 0x0A00;
	}

	*(screen + track + sector*80) = code;
}

//===========================================================================
//			      gaugeClear()
//===========================================================================

void gaugeClear(void)
{
	gaugePos = 1;
	textcolor(RED);textbackground(LIGHTGRAY);
	gotoxy(1, 20);cprintf(" SECTOR-METER ");
	gotoxy(1, 21);cprintf(" ------------ ");
}

//===========================================================================
//			       gaugePulse()
//===========================================================================

void gaugePulse(void)
{
	gotoxy(1 + gaugePos++, 21);
	textcolor(BLACK);textbackground(LIGHTGRAY);
	cputs("*");
}

//===========================================================================
//			       ReadDisk()
//===========================================================================

void ReadDisk(FILE *file)
{
	int done;

	MakeScreen();

	TrackZero();
	MotorOn();

	// set LPT port in input mode
	control |= LPT_INPUT_MODE_BIT;
	outportb(CONTROL_PORT, control);

	done = FALSE;

	while(!done)
	{
		// read side 0
		selectSide(0);
		ReadTrack();
		saveTrack(file);

		// read side 1
		selectSide(1);
		ReadTrack();
		saveTrack(file);

		// check for end of disk
		if((Track == 79) && (Side == 1))
		{
			sound(400); delay(70);
			sound(600); delay(70);
			sound(800); delay(70);
			nosound();
			done = TRUE;
		}

		TrackIn();

		if(kbhit())
		{
			if(getch() == 27)	// escape-key
				done = TRUE;
		}
	}

	MotorOff();
	TrackZero();

	/* deselect drive */
	control &= ~B_SELECT;
	outportb(CONTROL_PORT, control);
}

//===========================================================================
//			       PrintMessage()
//===========================================================================

void PrintMessage(char *msg)
{
	int lenght = strlen(msg);

	textbackground(BLUE);
	gotoxy(40 - (lenght/2), 10); textcolor(BLUE); cprintf("%s", msg);
	gotoxy(40 - (lenght/2), 11); textcolor(YELLOW); cprintf("%s", msg);
	gotoxy(40 - (lenght/2), 12); textcolor(BLUE); cprintf("%s", msg);

	gotoxy(30, 15); textcolor(LIGHTRED);
	cprintf("Press a key to continue");
	getch();

	MakeScreen();
}

//===========================================================================
//			   AskForFilename()
//===========================================================================

int AskForFilename(void)
{
	int lenght=0;
	char c;

	textcolor(YELLOW);textbackground(BLUE);
	gotoxy(28, 11);cprintf(" Please insert file name ");
	gotoxy(28, 12);cprintf("                         ");
	gotoxy(28, 13);cprintf("                         ");
	gotoxy(35, 13);textbackground(BLACK);cprintf(".adf        ");
	gotoxy(28, 14);textbackground(BLUE);cprintf("                         ");

	gotoxy(35, 13);
	_setcursortype(_NORMALCURSOR);

	for(;;)
	{
		gotoxy(35 + lenght, 13);
		c = getch();

		if(c == 8)	// delete
		{
			if(lenght)
			{
				strcpy(&filename[lenght-1],
							&filename[lenght]);
				lenght--;
				gotoxy(35, 13);textbackground(BLACK);
				cprintf("            ");
				gotoxy(35, 13);cprintf("%s", filename);
			}
		}

		if(c == 13)	// enter-key
			break;

		if(isalpha(c) || isdigit(c) || ( c == '-') || (c == '_'))
		{
			if(lenght >= 8)
				beep();
			else
			{
				filename[lenght] = c;
				lenght++;
				strcpy(&filename[lenght], ".adf");
				gotoxy(35, 13);textbackground(BLACK);
				cprintf("            ");
				gotoxy(35, 13);cprintf("%s", filename);
			}
		}
	}

	_setcursortype(_NOCURSOR);
	return(lenght);
}

//===========================================================================
//						AskForLPTPortAddress()
//===========================================================================

void AskForLPTPortAddress(void)
{
	int d=0;

	clrscr();

	puts("Choose LPT port address:");
	puts("------------------------");
	puts("\n1 - 0x378");
	puts("2 - 0x278");
	puts("3 - 0x3BC");
	puts("4 - 0x2BC");

	while((d<49) || (d>52))
		d = getch();

	if(d == 49) LPTBaseAddress=0x378;
	if(d == 50) LPTBaseAddress=0x278;
	if(d == 51) LPTBaseAddress=0x3BC;
	if(d == 52) LPTBaseAddress=0x2BC;
}

//===========================================================================
//  			        Main()
//===========================================================================

void main(void)
{
	FILE *file;

	_setcursortype(_NOCURSOR);

	AskForLPTPortAddress();
	MakeScreen();
	MotorOff();

	/* deselect drive */
	control &= ~B_SELECT;
	outportb(CONTROL_PORT, control);

	while(AskForFilename())
	{
		// check if a file with this name already exist
		file = fopen(filename, "rb");
		if(file)
		{
			fclose(file);
			PrintMessage("ATTENTION: a file with this name already exist !!");
		}

		else if(!(file = fopen(filename, "wb")))
			PrintMessage("ERROR: Unable to open file !!!!");
		else
		{
			ReadDisk(file);
			fclose(file);
		}
	}

	textbackground(BLACK);
	textcolor(LIGHTGRAY);
	clrscr();
	_setcursortype(_NORMALCURSOR);
	printf("\nAFR - Amiga Floppy Reader v1.0");
	printf("\n(c) 1997, Marco Veneri");
}
